// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       texture_piece.h
 *  \brief      テクスチャピース
 *  \date       Since: March 1, 2023. 5:36:09 JST.
 *  \author     Acerola
 *  \version    1.2.0
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_TEXTURE_PIECE_H_1677616569
#define INCGUARD_TEXTURE_PIECE_H_1677616569

#include "texture_atlas_holder.h"

#ifdef MGLEXT_TEXTURE_ATLAS_HOLDER_ENABLED

namespace MGLExt
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャピース
 *  \tparam     hashSeed    ハッシュ生成のシード値
 */
/* ------------------------------------------------------------------------- */
template <uint32_t hashSeed = MGL::Hash::kFNV1aDefaultValue32>
class TexturePiece
{
public:
    //! ハッシュのシード値
    static constexpr uint32_t kHashSeed = hashSeed;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      パスからハッシュ値を生成
     *  \param[in]  path    テクスチャのパス
     */
    /* ------------------------------------------------------------------------- */
    static constexpr uint32_t MakeHash(const char *path) noexcept
    {
        return MGL::Hash::FNV1a(path, kHashSeed);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  path    テクスチャのパス
     */
    /* ------------------------------------------------------------------------- */
    constexpr TexturePiece(const char *path) noexcept
        : TexturePiece(MakeHash(path))
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  hash    ハッシュ値
     */
    /* ------------------------------------------------------------------------- */
    constexpr TexturePiece(uint32_t hash) noexcept
        : _index(TextureAtlasHolder::GetInstance().GetIndex(hash))
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr TexturePiece() noexcept
        : _index(SIZE_MAX)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      テクスチャの範囲矩形を取得
     *  \return     テクスチャの範囲矩形
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const MGL::Rectangle &GetRectangle() const noexcept
    {
        if (const auto *location = TextureAtlasHolder::GetInstance().GetLocation(_index); location != nullptr)
        {
            return location->rectangle;
        }
        return _invalidRectangle;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      テクスチャを取得
     *  \return     テクスチャ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const MGL::Render::Texture &GetTexture() const noexcept
    {
        const auto &holder = TextureAtlasHolder::GetInstance();
        if (const auto *location = holder.GetLocation(_index); location != nullptr)
        {
            if (const auto *texture = holder.GetTexture(location->textureIndex); texture != nullptr)
            {
                return *texture;
            }
        }
        return _invalidTexture;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      bool型へのキャスト
     *  \retval     true    このオブジェクトは有効
     *  \retval     false   このオブジェクトは無効
     */
    /* ------------------------------------------------------------------------- */
    explicit constexpr operator bool() const noexcept
    {
        return _index < TextureAtlasHolder::GetInstance().GetLocationCount();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      否定演算子
     *  \retval     true    このオブジェクトは無効
     *  \retval     false   このオブジェクトは有効
     */
    /* ------------------------------------------------------------------------- */
    constexpr bool operator!() const noexcept
    {
        return !static_cast<bool>(*this);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      領域付きのテクスチャに変換
     *  \return     領域付きのテクスチャ
     */
    /* ------------------------------------------------------------------------- */
    operator MGL::Render::TextureWithBounds() const noexcept
    {
        return MGL::Render::TextureWithBounds(GetTexture(), GetRectangle());
    }

private:
    size_t _index;
    inline static MGL::Rectangle _invalidRectangle;
    inline static MGL::Render::Texture _invalidTexture;
};
}    // namespace MGLExt

#endif    // defined(MGLEXT_TEXTURE_ATLAS_HOLDER_ENABLED)
#endif    // INCGUARD_TEXTURE_PIECE_H_1677616569

// vim: et ts=4 sw=4 sts=4
