// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       texture_atlas_holder.h
 *  \brief      テクスチャアトラスホルダー
 *  \date       Since: June 22, 2021. 9:55:55 JST.
 *  \author     Acerola
 *  \version    1.2.0
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_TEXTURE_ATLAS_HOLDER_H_1624323355
#define INCGUARD_TEXTURE_ATLAS_HOLDER_H_1624323355

#include <mgl/mgl.h>

//! テクスチャアトラスホルダーの利用に必要なMGLのバージョン
#define MGLEXT_TEXTURE_ATLAS_HOLDER_REQUIRE_VERSION MGL_MAKE_VERSION(1, 1, 16)

// 必要バージョンのチェック
#if MGL_CURRENT_VERSION < MGLEXT_TEXTURE_ATLAS_HOLDER_REQUIRE_VERSION
#error This extension requires MGL version 1.1.16 or later.
#else    // MGL_CURRENT_VERSION < MGLEXT_TEXTURE_ATLAS_HOLDER_REQUIRE_VERSION

//! テクスチャアトラスホルダーが利用可能であることを示す定義
#define MGLEXT_TEXTURE_ATLAS_HOLDER_ENABLED

namespace MGLExt
{
//! テクスチャアトラスホルダー
class TextureAtlasHolder final : public MGL::StaticSingleton<TextureAtlasHolder>
{
public:
    //! ロケーション情報
    struct Location
    {
        uint32_t hash{0};            //!< ハッシュ値
        size_t textureIndex{0};      //!< テクスチャのインデックス
        MGL::Rectangle rectangle;    //!< テクスチャの範囲矩形
    };

    bool Load(const char *path, const char *name) noexcept;

    [[nodiscard]] size_t GetIndex(uint32_t hash) noexcept;
    [[nodiscard]] const Location *GetLocation(size_t index) const noexcept;
    [[nodiscard]] const MGL::Render::Texture *GetTexture(size_t index) const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ロケーションの数を取得
     *  \return     ロケーションの数
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr size_t GetLocationCount() const noexcept
    {
        return _locationCount;
    }

private:
    //! ロケーション情報のヘッダ
    struct LocationHeader
    {
        uint32_t hashSeed;         //!< ハッシュのシード値
        uint32_t identify;         //!< 識別子
        uint32_t revision;         //!< リビジョン情報
        uint32_t flags;            //!< フラグ情報
        uint32_t imageCount;       //!< 画像ファイルの数
        uint32_t locationCount;    //!< ロケーションの数
    };


    bool LoadLocations(const void *buffer, size_t bufferSize, const MGL::STL::string &filename) noexcept;
    bool LoadTexture(const char *path, const char *name) noexcept;

    MGL::STL::unique_ptr<Location[]> _locations{nullptr};
    MGL::STL::vector<MGL::Render::Texture> _textures;

    size_t _imageCount{0};
    size_t _locationCount{0};
};
}    // namespace MGLExt

#endif    // MGL_CURRENT_VERSION < MGLEXT_TEXTURE_ATLAS_HOLDER_REQUIRE_VERSION
#endif    // INCGUARD_TEXTURE_ATLAS_HOLDER_H_1624323355

// vim: et ts=4 sw=4 sts=4
