// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       texture_atlas_holder.cc
 *  \brief      テクスチャアトラスホルダー
 *  \date       Since: June 22, 2021. 9:55:54 JST.
 *  \author     Acerola
 *  \version    1.2.0
 */
/* ------------------------------------------------------------------------- */

#include "texture_atlas_holder.h"

#ifdef MGLEXT_TEXTURE_ATLAS_HOLDER_ENABLED

namespace MGLExt
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャとロケーションを読み込み
 *  \param[in]  path    読み込みパス
 *  \param[in]  name    読み込む名前
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool TextureAtlasHolder::Load(const char *path, const char *name) noexcept
{
    if (_locations != nullptr)
    {
        return false;
    }

    try
    {
        // パスの末尾に'/'が付いていない場合は付与
        MGL::STL::string directory(path);
        if (directory.back() != '/')
        {
            directory += '/';
        }

        const MGL::STL::string locationPath = directory + name + "_loc.bin";
        MGL::File::ThrowingHandle handle(locationPath.c_str());

        // ロケーションファイル用のバッファを用意
        const auto size = handle.GetSize();
        const auto buffer = MGL::STL::make_unique<std::byte[]>(size);
        if (buffer == nullptr)
        {
            return false;
        }

        // ロケーションファイルを読み込み
        handle.Read(buffer.get(), size);

        // ロケーションファイルを解析
        if (!LoadLocations(buffer.get(), size, locationPath))
        {
            return false;
        }

        // 各画像を読み込み
        if (!LoadTexture(directory.c_str(), name))
        {
            return false;
        }

        return true;
    }
    catch (MGL::File::Exception &fileException)
    {
        MGL_ERROR(fileException.what());
        return false;
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ロケーションを読み込み
 *  \param[in]  buffer      ロケーションの格納先バッファ
 *  \param[in]  bufferSize  バッファサイズ
 *  \param[in]  filename    ファイル名
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool TextureAtlasHolder::LoadLocations(const void *buffer, size_t bufferSize, [[maybe_unused]] const MGL::STL::string &filename) noexcept
{
    if (_locations != nullptr)
    {
        return false;
    }

    MGL::ByteStream stream(buffer, bufferSize);

    // ヘッダの読み込み
    LocationHeader header = {};
    stream.Read(header);

    // 識別子チェック
    if (header.identify != MGL::Hash::FNV1a("TextureAtlasLocations", header.hashSeed))
    {
        MGL_ERROR("[TextureAtlasHolder] %s is not texture location file.", filename.c_str());
        return false;
    }

    // リビジョンチェック
    if (header.revision != 0)
    {
        MGL_ERROR("[TextureAtlasHolder] %s is unsupported revision.", filename.c_str());
        return false;
    }

    // 画像ファイルとロケーションの数を取得
    _imageCount = static_cast<size_t>(header.imageCount);
    _locationCount = static_cast<size_t>(header.locationCount);

    // 各ロケーションの読み込み
    _locations = MGL::STL::make_unique<Location[]>(_locationCount);
    for (size_t i = 0; i < _locationCount; i++)
    {
        auto &rectangle = _locations[i];
        rectangle.hash = stream.Read<uint32_t>();
        rectangle.textureIndex = stream.Read<uint32_t>();
        rectangle.rectangle.x = static_cast<float>(stream.Read<uint32_t>());
        rectangle.rectangle.y = static_cast<float>(stream.Read<uint32_t>());
        rectangle.rectangle.width = static_cast<float>(stream.Read<uint32_t>());
        rectangle.rectangle.height = static_cast<float>(stream.Read<uint32_t>());
        stream.Skip(sizeof(uint32_t));
    }

    // 終了マーカーチェック
    auto endMarker = stream.Read<uint32_t>();
    if (endMarker != MGL::Hash::FNV1a("EndOfRecord", header.hashSeed))
    {
        MGL_ERROR("[TextureAtlasHolder] End marker did not match.");
        return false;
    }

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャを読み込み
 *  \param[in]  path    読み込みパス
 *  \param[in]  name    読み込む名前
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool TextureAtlasHolder::LoadTexture(const char *path, const char *name) noexcept
{
    for (size_t i = 0; i < _imageCount; i++)
    {
        const MGL::STL::string imagePath = MGL::STL::string(path) + name + std::to_string(i).c_str() + ".png";

        MGL::Render::Texture texture;
        texture.Load(imagePath.c_str());
        if (!texture)
        {
            return false;
        }

        _textures.push_back(std::move(texture));
    }

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ハッシュ値からインデックスを取得
 *  \param[in]  hash    ハッシュ値
 *  \return     ハッシュ値に対応したインデックス．見つからない場合はロケーション数
 */
/* ------------------------------------------------------------------------- */
size_t TextureAtlasHolder::GetIndex(uint32_t hash) noexcept
{
    size_t min = 0;
    size_t max = _locationCount;

    do
    {
        const size_t index = min + ((max - min) / 2);
        const auto value = _locations[index].hash;

        if (value == hash)
        {
            return index;
        }
        else if (value > hash)
        {
            max = index;
        }
        else if (value < hash)
        {
            min = index + 1;
        }

    } while (min != max);

    MGL_ERROR("[TextureAtlasHolder] No matching hash %08X", hash);

    return _locationCount;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ロケーションを取得
 *  \param[in]  index   インデックス
 *  \return     インデックスに対応したロケーション．無効なインデックスを指定した場合はnullptr
 */
/* ------------------------------------------------------------------------- */
const TextureAtlasHolder::Location *TextureAtlasHolder::GetLocation(size_t index) const noexcept
{
    if (index >= _locationCount)
    {
        return nullptr;
    }

    return &_locations[index];
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャを取得
 *  \param[in]  index   インデックス
 *  \return     インデックスに対応したテクスチャ．無効なインデックスを指定した場合はnullptr
 */
/* ------------------------------------------------------------------------- */
const MGL::Render::Texture *TextureAtlasHolder::GetTexture(size_t index) const noexcept
{
    if (index >= _imageCount)
    {
        return nullptr;
    }

    return &_textures[index];
}
}    // namespace MGLExt

#endif    // defined(MGLEXT_TEXTURE_ATLAS_HOLDER_ENABLED)

// vim: et ts=4 sw=4 sts=4
