// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       rasterized_font.h
 *  \brief      ラスタライズフォント
 *  \date       Since: June 22, 2021. 18:45:04 JST.
 *  \author     Acerola
 *  \version    1.3.0
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_RASTERIZED_FONT_H_1624355104
#define INCGUARD_RASTERIZED_FONT_H_1624355104

#include <mgl/mgl.h>

//! ラスタライズフォントの利用に必要なMGLのバージョン
#define MGLEXT_RASTERIZED_FONT_REQUIRE_VERSION MGL_MAKE_VERSION(1, 1, 16)

// 必要バージョンのチェック
#if MGL_CURRENT_VERSION < MGLEXT_RASTERIZED_FONT_REQUIRE_VERSION
#error This extension requires MGL version 1.1.16 or later.
#else    // MGL_CURRENT_VERSION < MGLEXT_RASTERIZED_FONT_REQUIRE_VERSION

//! ラスタライズフォントが利用可能であることを示す定義
#define MGLEXT_RASTERIZED_FONT_ENABLED

namespace MGLExt
{
//! ラスタライズフォントリソースクラス
class RasterizedFont : public MGL::Render::IndexedFontResource
{
public:
    RasterizedFont(const MGL::File::PathView &path, const char *name) noexcept;

    [[nodiscard]] size_t GetIndex(char32_t character, MGL::Render::FontFaceType faceType) const noexcept override;
    [[nodiscard]] const MGL::Render::FontGlyph *GetGlyph(size_t index, MGL::Render::FontFaceType faceType, const MGL::Render::FontOption &option) const noexcept override;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したフェイスを保持しているかを取得
     *  \param[in]  faceType    フェイスタイプ
     *  \retval     true        保持している
     *  \retval     false       保持していない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool HasFontFace(MGL::Render::FontFaceType faceType) const noexcept override
    {
        return _faces.count(faceType) != 0;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このフォントリソースが有効かどうかを取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsValid() const noexcept override
    {
        return _isValid;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      改行の送りサイズを取得
     *  \return     改行の送りサイズ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] float GetLineAdvance() const noexcept override
    {
        return _lineAdvance;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ルビのオフセット値を取得
     *  \return     ルビのオフセット値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] float GetRubyOffset() const noexcept override
    {
        return _rubyOffset;
    }

private:
    //! バイナリファイルのヘッダ
    struct BinaryHeader
    {
        alignas(4) uint32_t identify;        //!< 識別子
        alignas(4) uint32_t revision;        //!< リビジョン番号
        alignas(4) uint32_t faceCount;       //!< フォントフェイスの数
        alignas(4) uint32_t textureCount;    //!< テクスチャの数
        alignas(4) uint32_t maxHeight;       //!< 全ての文字の高さの最大値
        alignas(4) uint32_t flags;           //!< フラグ情報
    };
    static_assert(sizeof(BinaryHeader) == 24, "Invalid BinaryHeader size");

    //! バイナリファイルのチャンクヘッダ
    struct BinaryChunkHeader
    {
        alignas(4) uint32_t identify;     //!< 識別子
        alignas(4) uint32_t chunkSize;    //!< チャンクサイズ
    };
    static_assert(sizeof(BinaryChunkHeader) == 8, "Invalid BinaryChunkHeader size");

    //! フェイス情報
    struct FaceInfo
    {
        alignas(4) uint32_t faceType;         //!< フェイスタイプ
        alignas(4) uint32_t fontSize;         //!< フォントサイズ
        alignas(4) uint32_t glyphCount;       //!< グリフの数
        alignas(4) uint32_t alignmentSize;    //!< アライメントサイズ
        alignas(4) uint32_t maxHeight;        //!< 高さの最大値
    };
    static_assert(sizeof(FaceInfo) == 20, "Invalid FaceInfo size");

    //! グリフ情報
    struct GlyphInfo
    {
        alignas(4) char32_t code;             //!< 文字コード（UTF-32）
        alignas(4) uint32_t useCount;         //!< 使用回数
        alignas(2) uint16_t width;            //!< 幅
        alignas(2) uint16_t height;           //!< 高さ
        alignas(2) uint16_t advance;          //!< 送り幅
        alignas(2) int16_t bearingX;          //!< X方向のベアリング
        alignas(2) int16_t bearingY;          //!< Y方向のベアリング
        alignas(2) uint16_t textureX;         //!< テクスチャのX座標
        alignas(2) uint16_t textureY;         //!< テクスチャのY座標
        alignas(2) uint16_t textureWidth;     //!< テクスチャ上の幅
        alignas(2) uint16_t textureHeight;    //!< テクスチャ上の高さ
        alignas(2) uint16_t texturePage;      //!< テクスチャのページ
        alignas(4) uint32_t flags;            //!< フラグ情報
    };
    static_assert(sizeof(GlyphInfo) == 32, "Invalid GlyphInfo size");

    bool Load(const MGL::File::PathView &path, const char *name) noexcept;

    bool _isValid{false};
    BinaryHeader _fontInfo;
    MGL::STL::vector<MGL::Render::Texture> _textures;
    float _lineAdvance{0.0f};
    float _rubyOffset{0.0f};
    struct FontFace
    {
        FaceInfo info;
        MGL::STL::unique_ptr<MGL::Render::FontGlyph[]> glyphs;
    };
    MGL::STL::unordered_map<MGL::Render::FontFaceType, FontFace> _faces;
};
}    // namespace MGLExt

#endif    // MGL_CURRENT_VERSION < MGLEXT_RASTERIZED_FONT_REQUIRE_VERSION
#endif    // INCGUARD_RASTERIZED_FONT_H_1624355104

// vim: et ts=4 sw=4 sts=4
