// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       rasterized_font.cc
 *  \brief      ラスタライズフォント
 *  \date       Since: June 22, 2021. 18:45:02 JST.
 *  \author     Acerola
 *  \version    1.3.0
 */
/* ------------------------------------------------------------------------- */

#include "mglext/rasterized_font.h"

#ifdef MGLEXT_RASTERIZED_FONT_ENABLED

namespace MGLExt
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  path    フォントファイル一式があるディレクトリのパス
 *  \param[in]  name    フォント名
 */
/* ------------------------------------------------------------------------- */
RasterizedFont::RasterizedFont(const MGL::File::PathView &path, const char *name) noexcept
    : IndexedFontResource(MGL::Render::FontOrigin::BaseLine)
    , _fontInfo()
{
    _isValid = Load(path, name);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フォントの読み込み
 *  \param[in]  path    フォントファイル一式があるディレクトリのパス
 *  \param[in]  name    フォント名
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool RasterizedFont::Load(const MGL::File::PathView &path, const char *name) noexcept
{
    // メトリクス情報を読み込み
    try
    {
        const auto metricsFilePath = path.Append((MGL::STL::string(name) + "_metrics.bin").c_str());
        MGL::File::ThrowingHandle handle(metricsFilePath);

        // フォント情報を読み込み
        handle.Read(&_fontInfo, sizeof(BinaryHeader));

        // ヘッダの識別子をチェック
        if (_fontInfo.identify != MGL::Hash::FNV1a("FontMetricsV2"))
        {
            MGL_ERROR("[RasterizedFont] %s is not metrics file.", metricsFilePath.GetCString());
            return false;
        }

        // リビジョンチェック
        if (_fontInfo.revision != 0)
        {
            MGL_ERROR("[RasterizedFont] Metrics revision %s is not supported.", _fontInfo.revision);
            return false;
        }

        // テクスチャの読み込み
        for (uint32_t i = 0; i < _fontInfo.textureCount; i++)
        {
            MGL::Render::Texture texture;

            // Note: このパスの生成はもうちょっと工夫したいところ
            constexpr int kPathSize = 512;
            char texturePath[kPathSize];
            if (snprintf(texturePath, kPathSize, "%s_%02d.png", path.Append(name).GetCString(), i) >= kPathSize)
            {
                texturePath[kPathSize - 1] = '\0';
            }

            texture.Load(texturePath);
            if (!texture)
            {
                MGL_ERROR("[RasterizedFont] Failed to load texture: %s", texturePath);
                return false;
            }

            _textures.push_back(texture);
        }

        // 各チャンクを読み込み
        while (!handle.IsEOF())
        {
            BinaryChunkHeader chunkHeader = {};
            handle.Read(&chunkHeader, sizeof(BinaryChunkHeader));

            // フェイスチャンク
            if (chunkHeader.identify == MGL::Hash::FNV1a("FaceChunk"))
            {
                // フェイス情報を読み取り
                FaceInfo faceInfo = {};
                handle.Read(&faceInfo, sizeof(FaceInfo));

                // マップにフェイスを新規作成
                auto &face = _faces[MGL::Render::FontFaceType{faceInfo.faceType}];
                face.info = faceInfo;
                face.glyphs = MGL::STL::make_unique<MGL::Render::FontGlyph[]>(face.info.glyphCount);

                // グリフ情報を読み込み
                auto binaryGlyphs = MGL::STL::make_unique<GlyphInfo[]>(face.info.glyphCount);
                handle.Read(binaryGlyphs.get(), sizeof(GlyphInfo) * face.info.glyphCount);

                // 読み込んだグリフ情報からテーブルを構築
                for (size_t i = 0; i < face.info.glyphCount; i++)
                {
                    const auto &binGlyph = binaryGlyphs[i];
                    auto &glyph = face.glyphs[i];

                    glyph.code = binGlyph.code;
                    glyph.size = MGL::Vector2(static_cast<float>(binGlyph.width), static_cast<float>(binGlyph.height));
                    glyph.advance = static_cast<float>(binGlyph.advance);
                    glyph.bearing = MGL::Vector2(static_cast<float>(binGlyph.bearingX), static_cast<float>(binGlyph.bearingY));

                    if (binGlyph.texturePage < _fontInfo.textureCount)
                    {
                        glyph.texture = MGL::Render::TextureWithBounds(
                            _textures[binGlyph.texturePage],
                            MGL::Rectangle(
                                static_cast<float>(binGlyph.textureX),
                                static_cast<float>(binGlyph.textureY),
                                static_cast<float>(binGlyph.textureWidth),
                                static_cast<float>(binGlyph.textureHeight)));
                    }
                    else
                    {
                        glyph.texture = {};
                    }
                }
            }
            // 未対応のチャンクはスキップ
            else
            {
                handle.Skip(chunkHeader.chunkSize);
            }
        }
    }
    catch (const MGL::File::Exception &fileException)
    {
        MGL_ERROR(fileException.what());
        return false;
    }

    // 行の送りサイズをグリフの高さの最大値に設定
    _lineAdvance = static_cast<float>(_fontInfo.maxHeight);

    // ルビがあるなら追加で色々設定
    if (const auto &m = _faces.find(MGL::Render::FontFaceType::Ruby); m != _faces.end())
    {
        _rubyOffset = _lineAdvance;
        _lineAdvance += static_cast<float>(m->second.info.maxHeight);
        SetEnabled(MGL::Render::FontFeature::Ruby, true);
    }

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      グリフのインデックスを取得
 *  \param[in]  character   取得するUTF-32文字
 *  \param[in]  faceType    フェイスタイプ
 *  \return     対応したグリフのインデックス．見つからない場合は SIZE_T_MAX
 */
/* ------------------------------------------------------------------------- */
size_t RasterizedFont::GetIndex(char32_t character, MGL::Render::FontFaceType faceType) const noexcept
{
    const auto &m = _faces.find(faceType);
    if (m == _faces.end())
    {
        return SIZE_MAX;
    }
    const auto &face = m->second;

    size_t min = 0;
    size_t max = face.info.glyphCount;

    do
    {
        const size_t index = min + ((max - min) / 2);
        const auto value = face.glyphs[index].code;

        if (value == character)
        {
            return index;
        }
        else if (value > character)
        {
            max = index;
        }
        else if (value < character)
        {
            min = index + 1;
        }

    } while (min != max);

    return SIZE_MAX;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      グリフを取得
 *  \param[in]  index       取得するグリフのインデックス
 *  \param[in]  faceType    フェイスタイプ
 *  \param[in]  option      描画オプション
 *  \return     対応するグリフ情報．見つからない場合はnullptr
 */
/* ------------------------------------------------------------------------- */
const MGL::Render::FontGlyph *RasterizedFont::GetGlyph(
    size_t index,
    MGL::Render::FontFaceType faceType,
    [[maybe_unused]] const MGL::Render::FontOption &option) const noexcept
{
    const auto &m = _faces.find(faceType);
    if (m == _faces.end())
    {
        return nullptr;
    }
    const auto &face = m->second;

    if (index >= face.info.glyphCount)
    {
        return nullptr;
    }

    return &face.glyphs[index];
}

#endif    // defined(MGLEXT_RASTERIZED_FONT_ENABLED)
}    // namespace MGLExt
// vim: et ts=4 sw=4 sts=4
