/* ------------------------------------------------------------------------- */
/*!
 *  \file       message_holder.h
 *  \brief      メッセージホルダー
 *  \date       Since: June 29, 2021. 16:49:28 JST.
 *  \author     Acerola
 *  \version    1.2.0
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MESSAGE_HOLDER_H_1624952968
#define INCGUARD_MESSAGE_HOLDER_H_1624952968

#include <mgl/mgl.h>

//! メッセージホルダーの利用に必要なMGLのバージョン
#define MGLEXT_MESSAGE_HOLDER_REQUIRE_VERSION MGL_MAKE_VERSION(1, 1, 16)

// 必要バージョンのチェック
#if MGL_CURRENT_VERSION < MGLEXT_MESSAGE_HOLDER_REQUIRE_VERSION
#error This extension requires MGL version 1.1.16 or later.
#else    // MGL_CURRENT_VERSION < MGLEXT_MESSAGE_HOLDER_REQUIRE_VERSION

//! メッセージホルダーが利用可能であることを示す定義
#define MGLEXT_MESSAGE_HOLDER_ENABLED

namespace MGLExt
{
//! メッセージホルダークラス
class MessageHolder final : public MGL::StaticSingleton<MessageHolder>
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    MessageHolder() noexcept
        : _eventChangeLanguage(MGL::Event::NotifyType::ChangeLanguage, OnEventChangeLanguage, this)
    {
    }

    bool Load(const MGL::File::PathView &path) noexcept;

    bool SetLanguage(uint32_t languageHash) noexcept;
    bool SetLanguage(MGL::System::Language language) noexcept;

    [[nodiscard]] size_t GetIndex(uint32_t labelHash) const noexcept;

    [[nodiscard]] const char *GetMessage(size_t index) const noexcept;
    [[nodiscard]] const MGL::Text::IndexedCharacter *GetIndexedMessage(size_t index) const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このホルダーが保持しているラベルの数を取得
     *  \return     ラベルの数
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr size_t GetLabelCount() const noexcept
    {
        return static_cast<size_t>(_labelCount);
    }

    static void OnEventChangeLanguage(void *callbackArg, void *notifyArg) noexcept;

private:
    uint32_t _hashSeed{MGL::Hash::kFNV1aDefaultValue32};
    uint32_t _labelCount{0};
    uint32_t _languageCount{0};
    bool _isIndexed{false};
    MGL::STL::unique_ptr<uint32_t[]> _labelTable{nullptr};
    MGL::STL::unique_ptr<uint32_t[]> _languageTable{nullptr};
    MGL::STL::unique_ptr<uint32_t[]> _offsetPool{nullptr};
    MGL::STL::unique_ptr<std::byte[]> _messagePool{nullptr};

    size_t _currentLanguage{0};

    MGL::Event::Handle _eventChangeLanguage;
};
}    // namespace MGLExt

#endif    // MGL_CURRENT_VERSION < MGLEXT_MESSAGE_HOLDER_REQUIRE_VERSION
#endif    // INCGUARD_MESSAGE_HOLDER_H_1624952968

// vim: et ts=4 sw=4 sts=4
