// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       message_holder.cc
 *  \brief      メッセージホルダー
 *  \date       Since: June 29, 2021. 16:49:26 JST.
 *  \author     Acerola
 *  \version    1.2.0
 */
/* ------------------------------------------------------------------------- */

#include "message_holder.h"

#ifdef MGLEXT_MESSAGE_HOLDER_ENABLED

namespace MGLExt
{
namespace
{
// バイナリデータのビットフラグ
constexpr uint32_t kBinaryFlagIndexed = 1u << 0;    //!< インデックス化されているか
}    // namespace

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      メッセージデータの読み込み
 *  \param[in]  path    メッセージデータのパス
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool MessageHolder::Load(const MGL::File::PathView &path) noexcept
{
    try
    {
        MGL::File::ThrowingHandle file(path);

        // ハッシュのシード値を取得
        file.Read(&_hashSeed, 4);

        // ヘッダのチェック
        uint32_t hash = 0;
        file.Read(&hash, 4);
        if (hash != MGL::Hash::FNV1a("MessageData", _hashSeed))
        {
            MGL_ERROR("[MessageHolder] Not message data.");
            return false;
        }

        // リビジョンチェック
        uint32_t revision = 0;
        file.Read(&revision, sizeof(revision));
        if (revision != 0)
        {
            MGL_ERROR("[MessageHolder] Unsupported binary revision.");
            return false;
        }

        // 言語数とラベルの数を読み込み
        file.Read(&_languageCount, sizeof(_languageCount));
        file.Read(&_labelCount, sizeof(_labelCount));

        // フラグチェック
        uint32_t flags = 0;
        file.Read(&flags, sizeof(flags));
        _isIndexed = ((flags & kBinaryFlagIndexed) != 0);

        // 言語テーブルを読み込み
        const auto languageTableSize = sizeof(uint32_t) * _languageCount;
        _languageTable = MGL::STL::make_unique<uint32_t[]>(languageTableSize);
        file.Read(_languageTable.get(), languageTableSize);

        // ラベルテーブルを読み込み
        const auto labelTableSize = sizeof(uint32_t) * _labelCount;
        _labelTable = MGL::STL::make_unique<uint32_t[]>(labelTableSize);
        file.Read(_labelTable.get(), labelTableSize);

        // オフセットプールを読み込み
        uint32_t offsetPoolSize = 0;
        file.Read(&offsetPoolSize, sizeof(uint32_t));
        _offsetPool = MGL::STL::make_unique<uint32_t[]>(offsetPoolSize);
        file.Read(_offsetPool.get(), offsetPoolSize);

        // メッセージプールを読み込み
        uint32_t messagePoolSize = 0;
        file.Read(&messagePoolSize, sizeof(uint32_t));
        _messagePool = MGL::STL::make_unique<std::byte[]>(messagePoolSize);
        file.Read(_messagePool.get(), messagePoolSize);

        // 整合性チェック
        file.Read(&hash, sizeof(uint32_t));
        if (hash != MGL::Hash::FNV1a("EndOfRecord", _hashSeed))
        {
            MGL_ERROR("[MessageHolder] Consistency error.");
            return false;
        }
    }
    catch (const MGL::File::Exception &fileException)
    {
        MGL_ERROR(fileException.what());
        return false;
    }
    catch (MGL::STL::string &error)
    {
        MGL_ERROR("[MessageHolder] %s: %s", path, error.c_str());
        return false;
    }

    // 現在の言語設定を反映
    SetLanguage(MGL::System::Locale().GetLanguage());

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      使用言語の設定
 *  \param[in]  languageHash    使用言語のハッシュ値 
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool MessageHolder::SetLanguage(uint32_t languageHash) noexcept
{
    for (size_t i = 0; i < _languageCount; i++)
    {
        if (languageHash == _languageTable[i])
        {
            _currentLanguage = i;
            return true;
        }
    }

    return false;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      使用言語の設定
 *  \param[in]  language    使用言語
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool MessageHolder::SetLanguage(MGL::System::Language language) noexcept
{
    MGL::STL::string name;

    // clang-format off
    switch (language)
    {
        case MGL::System::Language::Arabic:                 name = "Arabic";                break;  //!< アラビア語
        case MGL::System::Language::BrazilPortuguese:       name = "BrazilPortuguese";      break;  //!< ブラジルポルトガル語
        case MGL::System::Language::Bulgarian:              name = "Bulgarian";             break;  //!< ブルガリア語
        case MGL::System::Language::Czech:                  name = "Czech";                 break;  //!< チェコ語
        case MGL::System::Language::Dansk:                  name = "Dansk";                 break;  //!< デンマーク語
        case MGL::System::Language::Deutsch:                name = "Deutsch";               break;  //!< ドイツ語
        case MGL::System::Language::English:                name = "English";               break;  //!< 英語
        case MGL::System::Language::French:                 name = "French";                break;  //!< フランス語
        case MGL::System::Language::Greek:                  name = "Greek";                 break;  //!< ギリシャ語
        case MGL::System::Language::Italiano:               name = "Italiano";              break;  //!< イタリア語
        case MGL::System::Language::Japanese:               name = "Japanese";              break;  //!< 日本語
        case MGL::System::Language::Korean:                 name = "Korean";                break;  //!< 韓国語
        case MGL::System::Language::LatinAmericaSpanish:    name = "LatinAmericaSpanish";   break;  //!< ラテンアメリカのスペイン語
        case MGL::System::Language::Magyar:                 name = "Magyar";                break;  //!< ハンガリー語
        case MGL::System::Language::Nederlands:             name = "Nederlands";            break;  //!< オランダ語
        case MGL::System::Language::Norsk:                  name = "Norsk";                 break;  //!< ノルウェー語
        case MGL::System::Language::Polski:                 name = "Polski";                break;  //!< ポーランド語
        case MGL::System::Language::Portuguese:             name = "Portuguese";            break;  //!< ポルトガル語
        case MGL::System::Language::Romanian:               name = "Romanian";              break;  //!< ルーマニア語
        case MGL::System::Language::Russian:                name = "Russian";               break;  //!< ロシア語
        case MGL::System::Language::SimplifiedChinese:      name = "SimplifiedChinese";     break;  //!< 簡体中文
        case MGL::System::Language::Spanish:                name = "Spanish";               break;  //!< スペイン語
        case MGL::System::Language::Suomi:                  name = "Suomi";                 break;  //!< フィンランド語
        case MGL::System::Language::Svenska:                name = "Svenska";               break;  //!< スウェーデン語
        case MGL::System::Language::Thai:                   name = "Thai";                  break;  //!< タイ語
        case MGL::System::Language::TraditionalChinese:     name = "TraditionalChinese";    break;  //!< 繁体中文
        case MGL::System::Language::Turkish:                name = "Turkish";               break;  //!< トルコ語
        case MGL::System::Language::Ukrainian:              name = "Ukrainian";             break;  //!< ウクライナ語
        case MGL::System::Language::Vietnamese:             name = "Vietnamese";            break;  //!< ベトナム語
        case MGL::System::Language::Invalid:                                                break;  //!< 無効な言語
        case MGL::System::Language::Unknown:                                                break;  //!< 不明
    };
    // clang-format on

    if (!SetLanguage(MGL::Hash::FNV1a(name.c_str(), _hashSeed)))
    {
        SetLanguage(_languageTable[0]);
        return false;
    }

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ラベルのハッシュ値からインデックスを取得
 *  \param[in]  labelHash   ラベルのハッシュ値
 *  \return     対応したインデックス．見つからない場合はラベル数より大きい数
 */
/* ------------------------------------------------------------------------- */
size_t MessageHolder::GetIndex(uint32_t labelHash) const noexcept
{
    size_t min = 0;
    size_t max = _labelCount;

    do
    {
        const size_t index = min + ((max - min) / 2);
        const auto value = _labelTable[index];

        if (value == labelHash)
        {
            return index;
        }

        if (value > labelHash)
        {
            max = index;
        }
        else if (value < labelHash)
        {
            min = index + 1;
        }

    } while (min != max);

    return _labelCount;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      メッセージの取得（UTF-8）
 *  \param[in]  index   インデックス
 *  \return     インデックスに対応したメッセージ．取得できない場合はnullptr
 */
/* ------------------------------------------------------------------------- */
const char *MessageHolder::GetMessage(size_t index) const noexcept
{
    if (_isIndexed)
    {
        return nullptr;
    }

    if ((index >= _labelCount) || (_currentLanguage >= _languageCount))
    {
        return nullptr;
    }

    const auto offset = _offsetPool[(index * _languageCount) + _currentLanguage] / sizeof(const char);
    const auto *pool = reinterpret_cast<const char *>(_messagePool.get());

    return &pool[offset];
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      メッセージの取得（インデックス文字列）
 *  \param[in]  index   インデックス
 *  \return     インデックスに対応したメッセージ．取得できない場合はnullptr
 */
/* ------------------------------------------------------------------------- */
const MGL::Text::IndexedCharacter *MessageHolder::GetIndexedMessage(size_t index) const noexcept
{
    if (!_isIndexed)
    {
        return nullptr;
    }

    if ((index >= _labelCount) || (_currentLanguage >= _languageCount))
    {
        return nullptr;
    }

    const auto offset = _offsetPool[(index * _languageCount) + _currentLanguage] / sizeof(MGL::Text::IndexedCharacter);
    const auto *pool = reinterpret_cast<MGL::Text::IndexedCharacter *>(_messagePool.get());

    return &pool[offset];
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      言語変更時のコールバック処理
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       更新後の言語
 */
/* ------------------------------------------------------------------------- */
void MessageHolder::OnEventChangeLanguage(void *callbackArg, void *notifyArg) noexcept
{
    auto *thisPtr = static_cast<MessageHolder *>(callbackArg);
    const auto language = *(static_cast<MGL::System::Language *>(notifyArg));

    thisPtr->SetLanguage(language);
}
}    // namespace MGLExt

#endif    // defined(MGLEXT_MESSAGE_HOLDER_ENABLED)

// vim: et ts=4 sw=4 sts=4
