// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_ui_widget.cc
 *  \brief      MGL UIウィジット
 *  \date       Since: July 5, 2021. 8:12:05 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/render/mgl_renderer_2d.h>
#include <mgl/ui/mgl_ui_event_context.h>
#include <mgl/ui/mgl_ui_widget.h>

namespace MGL::UI
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      更新処理
 *  \param[in]  offset      オフセット
 *  \param[in]  updateEvent イベント更新フラグ
 */
/* ------------------------------------------------------------------------- */
void Widget::Update(const MGL::Vector2 &offset, bool updateEvent) noexcept
{
    OnUpdate(GetAdjustedPosition() + offset);

    auto updatedPosition(GetAdjustedPosition() + offset);
    for (auto &child : _children)
    {
        child->Update(updatedPosition, false);
    }

    if (updateEvent)
    {
        UpdateEvent(_eventContext.get(), offset);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      イベントの更新処理
 *  \param[in]  eventContext    イベントコンテキスト
 *  \param[in]  offset          座標のオフセット値
 */
/* ------------------------------------------------------------------------- */
void Widget::UpdateEvent(EventContext *eventContext, const MGL::Vector2 &offset) noexcept
{
    // 可視状態でないウィジットはイベント更新を行わない
    if (!_isVisible)
    {
        return;
    }

    // このウィジットの座標を計算
    auto position(GetAdjustedPosition() + offset);

    // 子ウィジットのイベント更新を先に実行する
    for (auto it = _children.rbegin(); it != _children.rend(); ++it)
    {
        auto *child = *it;
        auto *context = child->_eventContext != nullptr ? child->_eventContext.get() : eventContext;
        child->UpdateEvent(context, position);
    }

    if (eventContext != nullptr)
    {
        // イベントIDが設定されているウィジットであればイベント対象として追加
        if (_eventIdentifier != kInvalidEventID)
        {
            const MGL::Rectangle rectangle(position, GetRectangle().GetSize());
            eventContext->AddTarget(EventTarget(this, rectangle));
        }

        // イベントコンテキストを持っているウィジットであればコンテキスト側のイベント更新処理を呼び出し
        if (eventContext == _eventContext.get())
        {
            if (!eventContext->GetTargets().empty())
            {
                auto result = eventContext->Update();

                if (result.type != EventType::None)
                {
                    for (auto &target : eventContext->GetTargets())
                    {
                        target.widget->ApplyEventResult(eventContext, result);
                    }
                }

                eventContext->ClearTargets();
            }
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      イベント結果の適用
 *  \param[in]  eventContext    イベントコンテキスト
 *  \param[in]  result          イベント結果
 */
/* ------------------------------------------------------------------------- */
void Widget::ApplyEventResult(EventContext *eventContext, const EventResult &result) noexcept
{
    switch (result.type)
    {
        // フォーカス
        case EventType::Focus:
            if (result.identifier == _eventIdentifier)
            {
                if (_eventState != EventState::Focusing)
                {
                    SendEvent(eventContext, EventType::Focus, result.argument, result.touchID);
                }
            }
            else if (_eventState == EventState::Focusing)
            {
                SendEvent(eventContext, EventType::Unfocus);
            }
            break;

        // フォーカス解除
        case EventType::Unfocus:
            if (result.identifier == _eventIdentifier)
            {
                if (_eventState == EventState::Focusing)
                {
                    SendEvent(eventContext, EventType::Unfocus, result.argument, result.touchID);
                }
            }
            break;

        // 決定
        case EventType::Decide:
            if (result.identifier == _eventIdentifier)
            {
                if (_eventState == EventState::None)
                {
                    SendEvent(eventContext, EventType::Focus, result.argument, result.touchID);
                }
                SendEvent(eventContext, EventType::Decide, result.argument, result.touchID);
            }
            else if (_eventState == EventState::Focusing)
            {
                SendEvent(eventContext, EventType::Unfocus);
            }
            break;

        // 決定
        case EventType::DecideAndUnfocus:
            if (result.identifier == _eventIdentifier)
            {
                SendEvent(eventContext, EventType::Decide, result.argument, result.touchID);
                SendEvent(eventContext, EventType::Unfocus, result.argument, result.touchID);
            }
            else if (_eventState == EventState::Focusing)
            {
                SendEvent(eventContext, EventType::Unfocus);
            }
            break;

        default:
            break;
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      イベントの送信処理
 *  \param[in]  eventContext    イベントコンテキスト
 *  \param[in]  eventType       送信するイベントの種類
 *  \param[in]  argument        イベント引数
 *  \param[in]  touchID         イベント発生時のタッチID
 */
/* ------------------------------------------------------------------------- */
void Widget::SendEvent(EventContext *eventContext, EventType eventType, uint32_t argument, MGL::Input::TouchID touchID) noexcept
{
    // フォーカスイベントなら状態をフォーカスにする
    if (eventType == EventType::Focus)
    {
        _eventState = EventState::Focusing;
    }
    // アンフォーカスイベントなら状態をなしにする
    else if (eventType == EventType::Unfocus)
    {
        _eventState = EventState::None;
    }

    // タッチIDはウィジットに保持
    _eventTouchID = touchID;

    // イベントコンテキストに送信
    eventContext->SendEvent({eventType, _eventIdentifier, argument, touchID});

    // ウィジットのイベント送信後の処理も呼び出し
    OnNotifiedEvent(eventType, argument);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      イベントコンテキストのアクティベート
 *  \param[in]  listener    イベントリスナー
 *  \param[in]  delegate    イベントデリゲート
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool Widget::ActivateEventContext(EventListener *listener, EventDelegate *delegate) noexcept
{
    if (_eventContext != nullptr)
    {
        return false;
    }

    _eventContext = STL::make_unique<EventContext>(listener, delegate);

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      イベントコンテキストのアクティベート
 *  \param[in]  listener    イベントリスナー
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool Widget::ActivateEventContext(EventListener *listener) noexcept
{
    if (_eventContext != nullptr)
    {
        return false;
    }

    _eventContext = STL::make_unique<EventContext>(listener);

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      描画処理
 *  \param[in]  offset  オフセット
 */
/* ------------------------------------------------------------------------- */
void Widget::Render(const MGL::Vector2 &offset) noexcept
{
    if (!IsVisible())
    {
        return;
    }

    OnRender(GetAdjustedPosition() + offset);

    auto updatedPosition(GetAdjustedPosition() + offset);
    for (auto &child : _children)
    {
        child->Render(updatedPosition);
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      可視状態の設定
 *  \param[in]  isVisible   設定する可視状態．trueで可視
 */
/* ------------------------------------------------------------------------- */
void Widget::SetVisible(bool isVisible) noexcept
{
    _isVisible = isVisible;

    // フォーカス中のウィジットを不可視に設定した場合はフォーカス解除のイベントを呼び出す
    if (!isVisible && (_eventState == EventState::Focusing))
    {
        auto *widget = this;
        while (widget != nullptr)
        {
            if (widget->_eventContext != nullptr)
            {
                SendEvent(widget->_eventContext.get(), EventType::Unfocus);
                break;
            }
            else
            {
                widget = widget->_parent;
            }
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      可視状態の取得
 *  \retval     true        可視
 *  \retval     false       不可視
 */
/* ------------------------------------------------------------------------- */
bool Widget::IsVisible() const noexcept
{
    const auto *widget = this;
    while (widget != nullptr)
    {
        if (!widget->_isVisible)
        {
            return false;
        }
        widget = widget->_parent;
    }

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      透過値の取得
 *  \return     透過値
 */
/* ------------------------------------------------------------------------- */
float Widget::GetTransparency() const noexcept
{
    float value = _transparency;

    auto *parent = _parent;
    while (parent != nullptr)
    {
        value *= parent->_transparency;
        parent = parent->_parent;
    }

    return value;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      親のサイズを取得
 *  \return     親のサイズ．親がいない場合はレンダーターゲットのサイズ
 */
/* ------------------------------------------------------------------------- */
MGL::Vector2 Widget::GetParentSize() const noexcept
{
    if (_parent == nullptr)
    {
        return MGL::Render::Renderer2D().GetRenderTarget().GetSize();
    }

    return _parent->GetRectangle().GetSize();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      位置を取得
 *  \return     ウィジットの位置
 */
/* ------------------------------------------------------------------------- */
MGL::Vector2 Widget::GetPosition() const noexcept
{
    auto position = GetAdjustedPosition();

    auto *widget = _parent;
    while (widget != nullptr)
    {
        position += widget->GetAdjustedPosition();
        widget = widget->_parent;
    }

    return position;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      子ウィジットの追加
 *  \param[in]  child   子ウィジット
 */
/* ------------------------------------------------------------------------- */
bool Widget::AddChild(Widget *child) noexcept
{
    if (child == nullptr)
    {
        return false;
    }

    _children.push_back(child);

    child->_parent = this;

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      子ウィジットの削除
 *  \param[in]  child   削除するウィジット
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool Widget::RemoveChild(Widget *child) noexcept
{
    if (child == nullptr)
    {
        return false;
    }

    for (auto it = _children.begin(); it != _children.end(); ++it)
    {
        if (*it == child)
        {
            _children.erase(it);
            return true;
        }
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      位置情報によって補正された座標を取得
 *  \return     補正後の座標
 */
/* ------------------------------------------------------------------------- */
MGL::Vector2 Widget::GetAdjustedPosition() const noexcept
{
    MGL::Vector2 parentSize;

    auto *parent = _parent;
    while (parent != nullptr)
    {
        parentSize = parent->GetRectangle().GetSize();
        if (!parentSize.IsZero())
        {
            break;
        }

        parent = parent->_parent;
    }
    if (parentSize.IsZero())
    {
        parentSize = MGL::Render::Renderer2D().GetRenderTarget().GetSize();
    }

    auto adjustedPosition = _pivotAlignment.AdjustRectangle(GetRectangle());
    adjustedPosition = _anchorAlignment.AdjustPosition(parentSize, adjustedPosition);

    return adjustedPosition;
}
}    // namespace MGL::UI

// vim: et ts=4 sw=4 sts=4
