// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_ui_event_default_delegate.cc
 *  \brief      MGL UIイベントのデフォルトデリゲート
 *  \date       Since: July 14, 2023. 16:29:27 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/gamepad/mgl_gamepad.h>
#include <mgl/input/keyboard/mgl_keyboard.h>
#include <mgl/input/mouse/mgl_mouse.h>
#include <mgl/input/touch/mgl_touch.h>
#include <mgl/ui/mgl_ui_event_default_delegate.h>
#include <mgl/ui/mgl_ui_widget.h>

namespace MGL::UI
{
namespace
{
//! カーソル移動アサインテーブル
constexpr struct
{
    UserInput input;             //!< 入力フラグ
    UserInput repeat;            //!< リピートフラグ
    Input::Keycode keycode;      //!< キーボードのキーコード
    Input::PadButton button1;    //!< ゲームパッドのボタン (1)
    Input::PadButton button2;    //!< ゲームパッドのボタン (2)
} kMoveAssignTable[] =
{
    {UserInput::MoveUp, UserInput::MoveUpHasARepeat, Input::Keycode::Up, Input::PadButton::Up, Input::PadButton::AnalogUp},
    {UserInput::MoveDown, UserInput::MoveDownHasARepeat, Input::Keycode::Down, Input::PadButton::Down, Input::PadButton::AnalogDown},
    {UserInput::MoveLeft, UserInput::MoveLeftHasARepeat, Input::Keycode::Left, Input::PadButton::Left, Input::PadButton::AnalogLeft},
    {UserInput::MoveRight, UserInput::MoveRightHasARepeat, Input::Keycode::Right, Input::PadButton::Right, Input::PadButton::AnalogRight},
};

//! 決定に割り当てるキーボードのキーコード
constexpr auto kDecideKeycode = Input::Keycode::Return;

//! 決定に割り当てるゲームパッドのボタン
constexpr auto kDecidePadButton = Input::PadButton::Decide;
}    // namespace


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      キー入力以外のUIイベントの更新処理（外部呼び出し用）
 *  \param[in]  targets         イベント対象のUIウィジット
 *  \param[in]  currentFocusID  現在フォーカス中のウィジットのイベントID
 *  \param[in]  scope           有効範囲
 *  \return     イベント更新結果
 */
/* ------------------------------------------------------------------------- */
EventResult DefaultEventDelegate::UpdateWithoutMove(const STL::vector<EventTarget> &targets,
                                                    EventID currentFocusID,
                                                    const InputScope &scope) noexcept
{
    auto result = UpdateMouseEvent(targets, currentFocusID, scope);
    if (result.type == EventType::None)
    {
        result = UpdateTouchEvent(targets, currentFocusID, scope);
    }

    return result;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      UIイベントの更新処理
 *  \param[in]  targets         イベント対象のUIウィジット
 *  \param[in]  currentFocusID  現在フォーカス中のウィジットのイベントID
 *  \return     イベント更新結果
 */
/* ------------------------------------------------------------------------- */
EventResult DefaultEventDelegate::OnUpdateUIEvent(const STL::vector<EventTarget> &targets,
                                                  EventID currentFocusID) noexcept
{
    // フォーカスの要求があればそれを優先
    if (_focusRequestID != kInvalidEventID)
    {
        auto focusID = _focusRequestID;
        _focusRequestID = kInvalidEventID;
        return {EventType::Focus, focusID};
    }

    auto input = OnUpdateUserInput();

    // キー入力
    if (currentFocusID != kInvalidEventID)
    {
        // 決定
        if (input.Has(UserInput::Decide))
        {
            return {EventType::Decide, currentFocusID};
        }
        // 移動
        else
        {
            if (_moveTable.count(currentFocusID) != 0)
            {
                auto result = UpdateKeyInput(input, _moveTable[currentFocusID]);
                if (result.type != EventType::None)
                {
                    return result;
                }
            }
        }
    }
    // 何も入力されていない状態では専用のテーブルを使用する
    else
    {
        auto result = UpdateKeyInput(input, _firstInputMoveTable);
        if (result.type != EventType::None)
        {
            return result;
        }
    }

    // マウスとタッチの処理
    if (_isEnabledMouseAndTouch)
    {
        return UpdateWithoutMove(targets, currentFocusID, _scope);
    }

    return {};
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      移動テーブルを設定
 *  \param[in]  target  対象となるウィジットのイベントID
 *  \param[in]  table   設定するテーブル
 */
/* ------------------------------------------------------------------------- */
void DefaultEventDelegate::SetMoveTable(EventID target, const EventMoveTable &table) noexcept
{
    if (_moveTable.count(target) == 0)
    {
        _moveTable.emplace(target, table);
    }
    else
    {
        _moveTable[target] = table;
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      移動ーブルを取得
 *  \param[in]  target  対象となるウィジットのイベントID
 *  \return     対応する移動テーブル
 *  \note
 *      テーブルが存在しない場合は新規に生成して結果を返す
 */
/* ------------------------------------------------------------------------- */
EventMoveTable &DefaultEventDelegate::GetMoveTable(EventID target) noexcept
{
    return _moveTable[target];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      移動テーブルを保持しているかを取得
 *  \param[in]  target  対象となるウィジットのイベントID
 *  \retval     true    移動テーブルを持っている
 *  \retval     false   移動テーブルは持っていない
 */
/* ------------------------------------------------------------------------- */
bool DefaultEventDelegate::HasMoveTable(EventID target) const noexcept
{
    return (_moveTable.count(target) != 0);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テーブルをクリア
 */
/* ------------------------------------------------------------------------- */
void DefaultEventDelegate::ClearTable() noexcept
{
    // 用途とアロケータの効率を考慮するとコンテナをクリアするより値を初期化したほうが良さそう
#if 0
    _moveTable.clear();
#else
    for (auto &table : _moveTable)
    {
        table.second = EventMoveTable();
    }
#endif

    _firstInputMoveTable = EventMoveTable();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ユーザー入力の更新処理
 *  \return     ユーザー入力フラグ
 */
/* ------------------------------------------------------------------------- */
UserInputFlags DefaultEventDelegate::OnUpdateUserInput() const noexcept
{
    UserInputFlags flags;

    const Input::Keyboard keyboard;
    const Input::Gamepad gamepad;

    // カーソル移動の更新
    for (const auto assign : kMoveAssignTable)
    {
        // キーボード
        if (keyboard.IsConnecting())
        {
            // リピートなし
            if (keyboard.IsTriggered(assign.keycode))
            {
                flags.Set(assign.input);
            }
            // リピートあり
            else if (keyboard.IsARepeat(assign.keycode))
            {
                flags.Set(assign.input);
                flags.Set(assign.repeat);
            }
        }

        // ゲームパッド
        if (gamepad)
        {
            // リピートなし
            if (gamepad.IsTriggered(assign.button1) || gamepad.IsTriggered(assign.button2))
            {
                flags.Set(assign.input);
            }
            // リピートあり
            else if (gamepad.IsARepeat(assign.button1) || gamepad.IsARepeat(assign.button2))
            {
                flags.Set(assign.input);
                flags.Set(assign.repeat);
            }
        }
    }

    // キーボードによる決定
    if (keyboard.IsConnecting())
    {
        if (keyboard.IsTriggered(kDecideKeycode))
        {
            flags.Set(UserInput::Decide);
        }
    }

    // ゲームパッドによる決定
    if (gamepad)
    {
        if (gamepad.IsTriggered(kDecidePadButton))
        {
            flags.Set(UserInput::Decide);
        }
    }

    return flags;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      キー入力イベントの更新処理
 *  \param[in]  input   ユーザー入力
 *  \param[in]  table   リンクテーブル
 *  \return     イベント結果
 */
/* ------------------------------------------------------------------------- */
EventResult DefaultEventDelegate::UpdateKeyInput(UserInputFlags input, const EventMoveTable &table) noexcept
{
    // チェック用のテーブル
    const struct
    {
        const EventMoveLink &link;
        UserInput input;
        UserInput repeat;
    } kCheckIDArray[] =
    {
        {table.up, UserInput::MoveUp, UserInput::MoveUpHasARepeat},
        {table.down, UserInput::MoveDown, UserInput::MoveDownHasARepeat},
        {table.left, UserInput::MoveLeft, UserInput::MoveLeftHasARepeat},
        {table.right, UserInput::MoveRight, UserInput::MoveRightHasARepeat},
    };

    for (const auto &check : kCheckIDArray)
    {
        if (check.link && input.Has(check.input))
        {
            // ループフラグが設定されている場合はリピート入力を無視する
            if (check.link.isLoop && input.Has(check.repeat))
            {
                continue;
            }

            return {MGL::UI::EventType::Focus, check.link.value};
        }
    }

    return {};
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウス入力の更新処理
 *  \param[in]  targets         イベント対象のUIウィジット
 *  \param[in]  currentFocusID  現在のフォーカスID
 *  \param[in]  scope           有効範囲
 *  \return     イベント更新結果
 */
/* ------------------------------------------------------------------------- */
EventResult DefaultEventDelegate::UpdateMouseEvent(const STL::vector<EventTarget> &targets, EventID currentFocusID, const InputScope &scope) noexcept
{
    const Input::Mouse mouse;

    // マウスが接続されていなければ処理しない
    if (!mouse.IsConnecting())
    {
        return {};
    }

    EventResult result;


    if (!mouse.GetDeltaMove().IsZero() || mouse.IsTriggered(Input::MouseButton::Left))
    {
        // 有効範囲外であれば処理しない
        if (!scope.IsValid())
        {
            if (!scope.IsEnclosed(mouse.GetPosition()))
            {
                if ((scope.GetBehavior() == ScopeOutBehavior::Unfocus) && (currentFocusID != kInvalidEventID))
                {
                    return {EventType::Unfocus, currentFocusID};
                }
                return result;
            }
        }

        for (const auto &target : targets)
        {
            // マウスカーソルがウィジットの範囲内に入っていればフォーカス or 決定
            if (target.rectangle.IsEnclosed(mouse.GetPosition()))
            {
                // 左クリックされた場合は決定を送信（フォーカスも自動で行われる）
                if (mouse.IsTriggered(Input::MouseButton::Left))
                {
                    return {EventType::Decide, target.widget->GetEventIdentifier()};
                }
                // 単にカーソルがウィジットの上に乗っているだけであればフォーカス
                else
                {
                    // 既にフォーカス中なら継続
                    if (target.widget->GetEventState() == EventState::Focusing)
                    {
                        return {};
                    }

                    return {EventType::Focus, target.widget->GetEventIdentifier()};
                }
            }
            // ウィジットがフォーカス中でカーソルが範囲外に出た場合はフォーカス解除
            else if (target.widget->GetEventState() == EventState::Focusing)
            {
                result = EventResult(EventType::Unfocus, target.widget->GetEventIdentifier());
            }
        }
    }

    return result;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タッチ入力の更新処理
 *  \param[in]  targets         イベント対象のUIウィジット
 *  \param[in]  currentFocusID  現在のフォーカスID
 *  \param[in]  scope           有効範囲
 *  \return     イベント更新結果
 */
/* ------------------------------------------------------------------------- */
EventResult DefaultEventDelegate::UpdateTouchEvent(const STL::vector<EventTarget> &targets, EventID currentFocusID, const InputScope &scope) noexcept
{
    // タッチデバイスが利用不可能な環境では処理しない
    if (!Input::Touch::IsAvailable())
    {
        return {};
    }

    // フォーカスなし
    if (currentFocusID == kInvalidEventID)
    {
        for (const auto &target : targets)
        {
            const auto *widget = target.widget;

            // 初回タッチ，または一度フォーカスしたタッチがウィジットの範囲内に入っている場合はフォーカスする
            if (const Input::Touch touch; touch.IsEnabled())
            {
                if (touch.IsTouchedFirst() || (touch.GetTouchID() == widget->GetEventTouchID()))
                {
                    const auto &position = touch.GetPosition();

                    // 有効範囲のチェック
                    if (scope.IsValid())
                    {
                        if (!scope.IsEnclosed(position))
                        {
                            continue;
                        }
                    }

                    // ウィジットの範囲内かをチェック
                    if (target.rectangle.IsEnclosed(position))
                    {
                        return {EventType::Focus, widget->GetEventIdentifier(), 0, touch.GetTouchID()};
                    }
                }
            }
        }
    }
    // フォーカス中（ウィジットをタッチ中）
    else
    {
        for (const auto &target : targets)
        {
            const auto *widget = target.widget;

            // フォーカス中のウィジットのみを処理
            if (widget->GetEventIdentifier() == currentFocusID)
            {
                // タッチ継続中
                if (const Input::Touch touch(widget->GetEventTouchID()); touch.IsEnabled())
                {
                    const auto &position = touch.GetPosition();

                    // 有効範囲のチェック
                    if (scope.IsValid())
                    {
                        if (!scope.IsEnclosed(position))
                        {
                            continue;
                        }
                    }

                    // タッチ位置がウィジットの範囲内であれば継続
                    if (target.rectangle.IsEnclosed(position))
                    {
                        return {};
                    }
                    // タッチ位置が範囲外に出たらフォーカス解除（再度範囲内に入った場合に備えてタッチIDは伝えておく）
                    else
                    {
                        return {EventType::Unfocus, widget->GetEventIdentifier(), 0, touch.GetTouchID()};
                    }
                }
                // フォーカス継続中にタッチが解除されたら決定
                else
                {
                    return {EventType::DecideAndUnfocus, widget->GetEventIdentifier(), 0, touch.GetTouchID()};
                }
            }
        }

        // フォーカス中のウィジットが見つからなければフォーカス解除
        return {EventType::Unfocus, currentFocusID};
    }

    return {};
}
}    // namespace MGL::UI

// vim: et ts=4 sw=4 sts=4
