// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_ui_event_context.cc
 *  \brief      MGL UIイベントコンテキスト
 *  \date       Since: July 7, 2023. 11:19:17 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/ui/mgl_ui_event_context.h>
#include <mgl/ui/mgl_ui_event_delegate.h>
#include <mgl/ui/mgl_ui_event_listener.h>

namespace MGL::UI
{
namespace
{
constexpr const size_t kTargetArrayReserveSize = 16;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  listener    イベントリスナー
 */
/* ------------------------------------------------------------------------- */
EventContext::EventContext(EventListener *listener) noexcept
    : _listener(listener)
    , _delegate(&_defaultDelegate)
{
    _targets.reserve(kTargetArrayReserveSize);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  listener    イベントリスナー
 *  \param[in]  delegate    イベントデリゲート
 */
/* ------------------------------------------------------------------------- */
EventContext::EventContext(EventListener *listener, EventDelegate *delegate) noexcept
    : _listener(listener)
    , _delegate(delegate)
{
    _targets.reserve(kTargetArrayReserveSize);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      イベントの更新
 *  \return     更新結果
 */
/* ------------------------------------------------------------------------- */
EventResult EventContext::Update() noexcept
{
    if (_delegate != nullptr)
    {
        return _delegate->OnUpdateUIEvent(_targets, _focusID);
    }

    return {};
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      イベントの送信
 *  \param[in]  result  送信するイベントの結果
 */
/* ------------------------------------------------------------------------- */
void EventContext::SendEvent(const EventResult &result) noexcept
{
    // リスナーのコールバック関数を呼び出し
    if (_listener != nullptr)
    {
        _listener->OnReceiveUIEvent(result);
    }

    // フォーカスイベントの送信であればそのIDを保持する
    if (result.type == EventType::Focus)
    {
        _focusID = result.identifier;
    }
    // アンフォーカスであればIDを初期化
    else if (result.type == EventType::Unfocus)
    {
        if (_focusID == result.identifier)
        {
            _focusID = kInvalidEventID;
        }
    }
}
}    // namespace MGL::UI

// vim: et ts=4 sw=4 sts=4
