// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_text_stream.cc
 *  \brief      テキストストリームクラス
 *  \date       Since: October 4, 2018. 2:13:17 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/text/mgl_text_stream.h>

#include <cstdio>
#include <cstdlib>

#include <mgl/file/mgl_file.h>
#include <mgl/system/mgl_system_debug_macro.h>

namespace MGL::Text
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  buffer      文字列バッファ
 *  \param[in]  size        バッファサイズ
 *  \param[in]  encoding    エンコードタイプ（省略で自動検出）
 *  \param[in]  existBOM    BOMが付与されているかのフラグ．エンコードタイプにAutoを指定した場合は参照しない
 */
/* ------------------------------------------------------------------------- */
TextStream::TextStream(const uint8_t *buffer, size_t size, Encoding encoding, bool existBOM) noexcept
{
    Set(buffer, size, encoding, existBOM);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      バッファの設定
 *  \param[in]  buffer      文字列バッファ
 *  \param[in]  size        バッファサイズ
 *  \param[in]  encoding    エンコードタイプ（省略で自動検出）
 *  \param[in]  existBOM    BOMが付与されているかのフラグ．エンコードタイプにAutoを指定した場合は参照しない
 */
/* ------------------------------------------------------------------------- */
void TextStream::Set(const uint8_t *buffer, size_t size, Encoding encoding, bool existBOM) noexcept
{
    _buffer = buffer;
    _size = size;
    _encoding = encoding;

    // エンコードタイプを検出
    if (_encoding == Encoding::Auto)
    {
        _encoding = CheckEncoding(_existBOM);
    }
    else
    {
        _existBOM = existBOM;
    }

    _offset = GetStartOffset();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルから読み込み
 *  \param[in]  filepath    ファイルのパス
 *  \param[in]  encoding    エンコードタイプ（省略で自動検出）
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool TextStream::Load(const File::PathView &filepath, Encoding encoding) noexcept
{
    // 読み込み
    try
    {
        File::ThrowingHandle handle(filepath);

        _size = handle.GetSize();
        _loadBuffer = STL::make_unique<uint8_t[]>(_size);
        handle.Read(_loadBuffer.get(), _size);
        _buffer = _loadBuffer.get();
        _encoding = encoding;
    }
    catch (const File::Exception &fileException)
    {
        MGL_ERROR(fileException.what());

        _size = 0;
        _loadBuffer.reset();
        _buffer = nullptr;

        return false;
    }

    // エンコードタイプを検出
    if (_encoding == Encoding::Auto)
    {
        _encoding = CheckEncoding(_existBOM);
    }

    _offset = GetStartOffset();

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字の取得
 *  \return     取得した文字
 */
/* ------------------------------------------------------------------------- */
Character TextStream::Read() noexcept
{
    Character character(0);

    if (_offset >= _size)
    {
        return character;
    }

    switch (_encoding)
    {
        case Encoding::UTF8:
            character = ReadUTF8();
            break;

        case Encoding::UTF16LE:
        case Encoding::UTF16BE:
            character = ReadUTF16();
            break;

        case Encoding::UTF32LE:
        case Encoding::UTF32BE:
            character = ReadUTF32();
            break;

        default:
            break;
    }

    return character;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字の取得（UTF-8）
 *  \return     取得した文字
 */
/* ------------------------------------------------------------------------- */
Character TextStream::ReadUTF8() noexcept
{
    Character character(0);

    if (_offset >= _size)
    {
        character.SetInvalidMarker();
        return character;
    }

    uint32_t code = _buffer[_offset++];
    int length = 0;

    // 1バイト
    if ((code & 0x80u) == 0)
    {
        length = 1;
    }
    // 2バイト
    else if ((code >> 5u) == 0x06)
    {
        code = code & 0x1Fu;
        length = 2;
    }
    // 3バイト
    else if ((code >> 4u) == 0x0E)
    {
        code = code & 0x0Fu;
        length = 3;
    }
    // 4バイト
    else if ((code >> 3u) == 0x1E)
    {
        code = code & 0x07u;
        length = 4;
    }
    // それ以外は不正
    else
    {
        character.SetCode(code);
        character.SetInvalidMarker();
        return character;
    }

    // 2バイト目以降を取得
    for (int i = 0; i < length - 1; i++)
    {
        if (_offset >= _size)
        {
            character.SetInvalidMarker();
            break;
        }

        const uint8_t lowCode = _buffer[_offset++];
        if ((lowCode >> 6u) != 0x02)
        {
            character.SetInvalidMarker();
        }

        code <<= 6u;
        code |= (lowCode & 0x3Fu);
    }

    // 最短で符号化されていない文字は不正扱い
    // このクラスでは常に4バイトで扱うため問題にはならないと思うけど，念のために．
    if (((length == 2) && (code <= 0x7F)) ||
        ((length == 3) && (code <= 0x7FF)) ||
        ((length == 4) && (code <= 0xFFFF)))
    {
        character.SetInvalidMarker();
    }

    character.SetCode(code);

    return character;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字の取得（UTF-16）
 *  \return     取得した文字
 */
/* ------------------------------------------------------------------------- */
Character TextStream::ReadUTF16() noexcept
{
    Character character(0);

    if ((_size - _offset) < 2)
    {
        character.SetInvalidMarker();
    }
    else
    {
        uint32_t code = 0;

        // 最初の2バイトを取得
        if (_encoding == Encoding::UTF16LE)
        {
            code = static_cast<uint32_t>(_buffer[_offset++]);
            code |= static_cast<uint32_t>(_buffer[_offset++] << 8u);
        }
        else if (_encoding == Encoding::UTF16BE)
        {
            code = static_cast<uint32_t>(_buffer[_offset++] << 8u);
            code |= static_cast<uint32_t>(_buffer[_offset++]);
        }

        // サロゲートペアであれば次の2バイトも取得
        if ((code >= 0xD800) && (code <= 0xDBFF))
        {
            uint32_t lowCode = 0;

            if (_encoding == Encoding::UTF16LE)
            {
                lowCode = static_cast<uint32_t>(_buffer[_offset++]);
                lowCode |= static_cast<uint32_t>(_buffer[_offset++] << 8u);
            }
            else if (_encoding == Encoding::UTF16BE)
            {
                lowCode = static_cast<uint32_t>(_buffer[_offset++] << 8u);
                lowCode |= static_cast<uint32_t>(_buffer[_offset++]);
            }

            // サロゲートペアの後半がフォーマットに基づいていなければ不正として扱う
            if ((lowCode >= 0xDC00) && (lowCode <= 0xDFFF))
            {
                character.SetInvalidMarker();
            }

            code = 0x10000 + ((code - 0xD800) * 0x400) + (lowCode - 0xDC00);
        }

        character.SetCode(code);
    }

    return character;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字の取得（UTF-32）
 *  \return     取得した文字
 */
/* ------------------------------------------------------------------------- */
Character TextStream::ReadUTF32() noexcept
{
    Character character(0);

    if ((_size - _offset) < 4)
    {
        character.SetInvalidMarker();
    }
    else
    {
        uint32_t code = 0;

        if (_encoding == Encoding::UTF32LE)
        {
            code = static_cast<uint32_t>(_buffer[_offset++]);
            code |= static_cast<uint32_t>(_buffer[_offset++] << 8u);
            code |= static_cast<uint32_t>(_buffer[_offset++] << 16u);
            code |= static_cast<uint32_t>(_buffer[_offset++] << 24u);
        }
        else
        {
            code = static_cast<uint32_t>(_buffer[_offset++] << 24u);
            code |= static_cast<uint32_t>(_buffer[_offset++] << 16u);
            code |= static_cast<uint32_t>(_buffer[_offset++] << 8u);
            code |= static_cast<uint32_t>(_buffer[_offset++]);
        }

        character.SetCode(code);
    }

    return character;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      情報の表示
 */
/* ------------------------------------------------------------------------- */
void TextStream::PrintInfo() const noexcept
{
    static const char * const typeTable[] =
    {
        "Auto",         // 自動認識
        "UTF-8",        // UTF-8
        "UTF-16 BE",    // UTF-16 ビッグエンディアン
        "UTF-16 LE",    // UTF-16 リトルエンディアン
        "UTF-32 BE",    // UTF-32 ビッグエンディアン
        "UTF-32 LE",    // UTF-32 リトルエンディアン
    };

    printf("Type: %s\n", typeTable[static_cast<int>(_encoding)]);
    printf("BOM:  %s\n", _existBOM ? "Yes" : "No");
    printf("Size: %zd Byte.\n", _size);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      全ての文字を文字配列に格納
 *  \param[out] array   文字配列の格納先
 *  \return     新たに格納した文字の数
 */
/* ------------------------------------------------------------------------- */
int TextStream::Read(CharacterArray &array) noexcept
{
    int addCount = 0;

    for (auto character = Read();; character = Read())
    {
        if (!character.IsValid())
        {
            break;
        }

        if (character.IsNull())
        {
            break;
        }

        array.push_back(character);
        addCount++;
    }

    return addCount;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      現在位置から行末まで読み込み
 *  \param[out] array   文字配列の格納先
 *  \return     新たに格納した文字の数
 */
/* ------------------------------------------------------------------------- */
int TextStream::ReadLine(CharacterArray &array) noexcept
{
    int addCount = 0;

    for (auto character = Read();; character = Read())
    {
        if (!character.IsValid())
        {
            break;
        }

        if (character.IsNewLine() || character.IsNull())
        {
            break;
        }

        array.push_back(character);
        addCount++;
    }

    return addCount;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      エンコードタイプを検査
 *  \param[out] existBOM  BOM付きかどうかのフラグの格納先
 *  \return     自動検出したエンコードタイプ．判別できなければUTF-8扱い．
 */
/* ------------------------------------------------------------------------- */
Encoding TextStream::CheckEncoding(bool &existBOM) const noexcept
{
    // UTF-8 BOM
    if (_size >= 3)
    {
        if ((_buffer[0] == 0xEF) && (_buffer[1] == 0xBB) && (_buffer[2] == 0xBF))
        {
            existBOM = true;
            return Encoding::UTF8;
        }
    }
    // UTF-32 BOM
    if (_size >= 4)
    {
        // LE
        if ((_buffer[0] == 0xFF) && (_buffer[1] == 0xFE) &&
            (_buffer[2] == 0x00) && (_buffer[3] == 0x00))
        {
            existBOM = true;
            return Encoding::UTF32LE;
        }

        // BE
        if ((_buffer[0] == 0x00) && (_buffer[1] == 0x00) &&
            (_buffer[2] == 0xFE) && (_buffer[3] == 0xFF))
        {
            existBOM = true;
            return Encoding::UTF32BE;
        }
    }
    // UTF-16 BOM
    if (_size >= 2)
    {
        // LE
        if ((_buffer[0] == 0xFF) && (_buffer[1] == 0xFE))
        {
            existBOM = true;
            return Encoding::UTF16LE;
        }
        // BE
        if ((_buffer[0] == 0xFE) || (_buffer[1] == 0xFF))
        {
            existBOM = true;
            return Encoding::UTF16BE;
        }
    }

    // BOMなしはUTF-8扱いにする
    existBOM = false;
    return Encoding::UTF8;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      開始オフセットを取得
 *  \return     開始オフセット
 */
/* ------------------------------------------------------------------------- */
size_t TextStream::GetStartOffset() const noexcept
{
    // BOM付きならエンコードタイプによって異なる
    if (_existBOM)
    {
        switch (_encoding)
        {
            // UTF-8
            case Encoding::UTF8:
                return 3;

            // UTF-16
            case Encoding::UTF16LE:
            case Encoding::UTF16BE:
                return 2;

            // UTF-32
            case Encoding::UTF32LE:
            case Encoding::UTF32BE:
                return 4;

            default:
                break;
        }
    }

    // 基本的には0
    return 0;
}

}    // namespace MGL::Text


// vim: et ts=4 sw=4 sts=4

