// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_text_index_converter.cc
 *  \brief      MGL テキストのインデックス化コンバータ
 *  \date       Since: June 27, 2021. 5:34:27 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <cstring>

#include <mgl/text/mgl_text_index_converter.h>

#include <mgl/common/mgl_color_xterm.h>
#include <mgl/stl/mgl_stl_containers.h>
#include <mgl/text/mgl_text_format_argument.h>

namespace MGL::Text
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字列をインデックス化
 *  \param[in]  text            文字列
 *  \param[in]  enableFormat    テキスト整形を有効にするかのフラグ
 *  \param[in]  enableTag       タグを有効にするかのフラグ
 *  \return     インデックス化された文字列．失敗時にはnullptr
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<IndexedCharacter[]> IndexConverter::ToIndexedString(const char *text, bool enableFormat, bool enableTag) const noexcept
{
    if (text == nullptr)
    {
        return nullptr;
    }

    STL::vector<Text::IndexedCharacter> indexes;
    indexes.reserve(strlen(text));

    auto currentFaceType = Render::FontFaceType::Default;

    const char *current = text;

    while (*current != '\0')
    {
        const char *next = nullptr;

        // 改行
        if ((*current == 0x0A) || (*current == 0x0D))
        {
            if (*current == 0x0A)    // LFのみ変換（CRは無視）
            {
                indexes.push_back(Text::kIndexedCharacterNewLine);
            }
            current++;
        }
        // 置換
        else if (next = ParseSubstitute(indexes, enableFormat, current, currentFaceType); next != current)
        {
            current = next;
        }
        // タグ
        else if (next = ParseTag(indexes, currentFaceType, enableTag, current); next != current)
        {
            current = next;
        }
        // それ以外はインデックスに変換
        else
        {
            char32_t u32char = 0;
            if (next = GetUTF32(u32char, current); next != nullptr)
            {
                auto index = ToIndexedCharacter(u32char, currentFaceType);
                if ((index != kIndexedCharacterInvalid) && (index != kIndexedCharacterIgnore))
                {
                    indexes.push_back(index);
                }
                current = next;
            }
            else
            {
                break;
            }
        }
    }

    indexes.push_back(Text::kIndexedCharacterEndOfText);

    // ユニークポインタに変換
    auto indexedString = STL::make_unique<Text::IndexedCharacter[]>(indexes.size());
    memcpy(indexedString.get(), indexes.data(), sizeof(Text::IndexedCharacter) * indexes.size());

    return indexedString;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      置換オプションを解析
 *  \param[out] indexes     インデックス文字の格納先
 *  \param[in]  enable      有効フラグ
 *  \param[in]  text        解析するテキスト
 *  \param[in]  faceType    フォントフェイスの種類
 *  \return     解析した文字の次のアドレス．失敗時はtextと同じ
 */
/* ------------------------------------------------------------------------- */
const char *IndexConverter::ParseSubstitute(STL::vector<Text::IndexedCharacter> &indexes,
                                            bool enable,
                                            const char *text,
                                            Render::FontFaceType faceType) const noexcept
{
    // フォーマットの無効化が指定されているならパースしない
    if (!enable)
    {
        return text;
    }

    // 先頭が'{'でなければ置換ではない
    if (text[0] != '{')
    {
        return text;
    }

    // '{'が連続して続いている場合はエスケープシーケンス
    if (text[1] == '{')
    {
        auto index = ToIndexedCharacter('{', faceType);
        if ((index != kIndexedCharacterInvalid) && (index != kIndexedCharacterIgnore))
        {
            indexes.push_back(index);
        }

        return &text[2];
    }

    // パースしてインデックス文字に変換
    FormatOptions options;
    if (const auto *next = FormatArgument::Parse(options, text); text != next)
    {
        if (options.isValid)
        {
            auto indexedOptions = Text::FormatArgument::ToIndexedCharacters(options);
            for (const auto index : indexedOptions)
            {
                indexes.push_back(index);
            }
            indexes.push_back(Text::kIndexedCharacterSubstitute);
        }

        return next;
    }
    return text;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          タグの解析
 *  \param[out]     indexes     インデックス文字の格納先
 *  \param[in,out]  faceType    フォントフェイスの種類
 *  \param[in]      enable      有効フラグ
 *  \param[in]      text        解析するテキスト
 *  \return         解析した文字の次のアドレス．失敗時はtextと同じ
 */
/* ------------------------------------------------------------------------- */
const char *IndexConverter::ParseTag(STL::vector<Text::IndexedCharacter> &indexes,
                                     Render::FontFaceType &faceType,
                                     bool enable,
                                     const char *text) const noexcept
{
    // 無効なら何もしない
    if (!enable)
    {
        return text;
    }

    // 先頭が'<'でなければタグではない
    const auto *current = text;
    if (*current != '<')
    {
        return text;
    }
    current++;

    // '<'が連続しているならエスケープシーケンス
    if (*current == '<')
    {
        auto index = ToIndexedCharacter('<', faceType);
        if ((index != kIndexedCharacterInvalid) && (index != kIndexedCharacterIgnore))
        {
            indexes.push_back(index);
        }

        return current + 1;
    }

    auto tagType = *current++;

    // タグを解析
    if ((tagType == '#') || (tagType == '$') || (tagType == '@') || (tagType == '/'))
    {
        STL::string tagName;
        tagName.reserve(96);

        if (const auto *next = GetTagName(tagName, current); next != current)
        {
            // 色指定
            if (tagType == '#')
            {
                ParseColorTag(indexes, tagName);
            }
            // 書体指定
            else if (tagType == '$')
            {
                ParseFaceTag(indexes, faceType, tagName);
            }
            // ルビ
            else if (tagType == '@')
            {
                ParseRuby(indexes, faceType, tagName);
            }
            // タグのリセット
            else if (tagType == '/')
            {
                ParseResetTag(indexes, faceType, tagName);
            }

            return next;
        }
    }

    return text;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          色指定タグを解析
 *  \param[out]     indexes     インデックス文字の格納先
 *  \param[in]      tagName     タグ名
 */
/* ------------------------------------------------------------------------- */
void IndexConverter::ParseColorTag(STL::vector<Text::IndexedCharacter> &indexes,
                                   const STL::string &tagName) noexcept
{
    if (!tagName.empty())
    {
        if (auto xcolor = GetXColor(Hash::FNV1a(tagName.c_str())); xcolor)
        {
            indexes.push_back(kTagParameterColor | static_cast<uint8_t>(xcolor.value()));
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          書体指定タグを解析
 *  \param[out]     indexes     インデックス文字の格納先
 *  \param[in,out]  faceType    フォントフェイスの種類
 *  \param[in]      tagName     タグ名
 */
/* ------------------------------------------------------------------------- */
void IndexConverter::ParseFaceTag(STL::vector<Text::IndexedCharacter> &indexes,
                                  Render::FontFaceType &faceType,
                                  const STL::string &tagName) const noexcept
{
    // 一旦はボールドのみに対応する．今後どうするかは未定
    if (tagName == "bold")
    {
        if (HasFontFace(Render::FontFaceType::Bold))
        {
            indexes.push_back(kTagParameterFace | static_cast<uint8_t>(FaceTypeIndex::Bold));
            faceType = Render::FontFaceType::Bold;
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ルビを解析
 *  \param[out]     indexes     インデックス文字の格納先
 *  \param[in]      faceType    フォントフェイスの種類
 *  \param[in]      tagName     タグ名
 *  \retval         true        成功
 *  \retval         false       失敗
 */
/* ------------------------------------------------------------------------- */
bool IndexConverter::ParseRuby(STL::vector<Text::IndexedCharacter> &indexes,
                               Render::FontFaceType faceType,
                               const STL::string &tagName) const noexcept
{
    STL::string baseText;
    STL::string rubyText;
    bool isWide = false;

    baseText.reserve(96);
    rubyText.reserve(96);

    auto hasRubyFace = HasFontFace(Render::FontFaceType::Ruby);

    const char *name = tagName.c_str();

    // 親文字を取得
    size_t delimiter = 0;
    for (size_t i = 0;; i++)
    {
        // 区切り文字':'をチェック
        if (name[i] == ':')
        {
            // ':'が連続していなければ区切り文字
            if (name[i + 1] != ':')
            {
                delimiter = i;
                break;
            }

            // 連続している場合はエスケープシーケンス
            baseText.push_back(':');
            i++;
        }
        // 区切り文字'='をチェック
        else if (name[i] == '=')
        {
            // '='が連続していなければ区切り文字
            if (name[i + 1] != '=')
            {
                isWide = true;
                delimiter = i;
                break;
            }

            // 連続している場合はエスケープシーケンス
            baseText.push_back('=');
            i++;
        }
        // 区切り文字が来る前にタグ名が終了していたらルビ扱いにしない
        else if (name[i] == '\0')
        {
            return false;
        }
        else
        {
            baseText.push_back(name[i]);
        }
    }

    // ルビ文字を取得
    for (size_t i = delimiter + 1; name[i] != '\0'; i++)
    {
        rubyText.push_back(name[i]);
    }

    // 親文字とルビ文字のどちらかが空であれば失敗
    if (baseText.empty() || rubyText.empty())
    {
        return false;
    }

    // 親文字をUTF-32に変換
    STL::u32string u32BaseText;
    u32BaseText.reserve(224);
    {
        const auto *current = baseText.c_str();
        while (*current != '\0')
        {
            char32_t u32char = 0;
            if (const auto *next = GetUTF32(u32char, current); next != nullptr)
            {
                u32BaseText.push_back(u32char);
                current = next;
            }
            else
            {
                return false;
            }
        }
    }

    // ルビ文字をUTF-32に変換
    STL::u32string u32RubyText;
    if (hasRubyFace)
    {
        u32RubyText.reserve(224);
        const auto *current = rubyText.c_str();
        while (*current != '\0')
        {
            char32_t u32char = 0;
            if (const auto *next = GetUTF32(u32char, current); next != nullptr)
            {
                u32RubyText.push_back(u32char);
                current = next;
            }
            else
            {
                return false;
            }
        }
    }

    // ルビ開始文字を挿入
    if (hasRubyFace)
    {
        indexes.push_back(isWide ? kIndexedCharacterWideRubyStart : kIndexedCharacterRubyStart);
    }

    // 親文字をインデックス文字に変換
    for (auto character : u32BaseText)
    {
        auto index = ToIndexedCharacter(character, faceType);
        if ((index != kIndexedCharacterInvalid) && (index != kIndexedCharacterIgnore))
        {
            indexes.push_back(index);
        }
    }

    if (hasRubyFace)
    {
        // ルビ区切り文字を挿入
        indexes.push_back(kIndexedCharacterRubyDelimiter);

        // ルビ文字をインデックス文字に変換
        for (auto character : u32RubyText)
        {
            auto index = ToIndexedCharacter(character, Render::FontFaceType::Ruby);
            if ((index != kIndexedCharacterInvalid) && (index != kIndexedCharacterIgnore))
            {
                indexes.push_back(index);
            }
        }

        // ルビ終了文字を挿入
        indexes.push_back(kIndexedCharacterRubyEnd);
    }

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          リセットタグを解析
 *  \param[out]     indexes     インデックス文字の格納先
 *  \param[in,out]  faceType    フォントフェイスの種類
 *  \param[in]      tagName     タグ名
 */
/* ------------------------------------------------------------------------- */
void IndexConverter::ParseResetTag(STL::vector<Text::IndexedCharacter> &indexes,
                                   Render::FontFaceType &faceType,
                                   const STL::string &tagName) noexcept
{
    // タグ名が空であれば全てのタグをリセット
    if (tagName.empty())
    {
        indexes.push_back(kIndexedCharacterTagReset);
        faceType = Render::FontFaceType::Default;
    }
    else
    {
        for (auto letter : tagName)
        {
            // 色変更をリセット
            if (letter == '#')
            {
                indexes.push_back(kIndexedCharacterColorReset);
            }
            // 書体変更をリセット
            else if (letter == '$')
            {
                indexes.push_back(kIndexedCharacterFaceReset);
                faceType = Render::FontFaceType::Default;
            }
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タグ名を取得
 *  \param[out] tagName     タグ名の格納先
 *  \param[in]  text        解析元のテキスト
 *  \return     解析後の次の文字の先頭アドレス．失敗時はtextと同じ
 */
/* ------------------------------------------------------------------------- */
const char *IndexConverter::GetTagName(STL::string &tagName, const char *text) noexcept
{
    for (size_t i = 0;; i++)
    {
        auto letter = text[i];

        if (letter == '>')
        {
            return &text[i + 1];
        }

        switch (letter)
        {
            case '\0':
            case 0x0D:
            case 0x0A:
                return text;

            default:
                tagName.push_back(letter);
                break;
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テキストからUTF-32文字を取得
 *  \param[out] u32char     UTF-32文字の格納先
 *  \param[in]  text        変換元のテキスト
 *  \return     解析した次の文字の先頭アドレス．失敗時はnullptr
 */
/* ------------------------------------------------------------------------- */
const char *IndexConverter::GetUTF32(char32_t &u32char, const char *text) noexcept
{
    u32char = static_cast<char32_t>(static_cast<uint8_t>(text[0]));
    int length = 0;

    // 1バイト
    if ((u32char & 0x80u) == 0)
    {
        length = 1;
    }
    // 2バイト
    else if ((u32char >> 5u) == 0x06)
    {
        u32char = u32char & 0x1Fu;
        length = 2;
    }
    // 3バイト
    else if ((u32char >> 4u) == 0x0E)
    {
        u32char = u32char & 0x0Fu;
        length = 3;
    }
    // 4バイト
    else if ((u32char >> 3u) == 0x1E)
    {
        u32char = u32char & 0x07u;
        length = 4;
    }
    // それ以外は不正
    else
    {
        return nullptr;
    }

    // 2バイト目以降を取得
    for (int i = 0; i < length - 1; i++)
    {
        auto lowCode = static_cast<uint8_t>(text[i + 1]);
        if ((lowCode >> 6u) != 0x02)
        {
            return nullptr;
        }

        u32char <<= 6u;
        u32char |= (lowCode & 0x3Fu);
    }

    // 最短で符号化されていない文字は不正扱い
    if (((length == 2) && (u32char <= 0x7F)) ||
        ((length == 3) && (u32char <= 0x7FF)) ||
        ((length == 4) && (u32char <= 0xFFFF)))
    {
        return nullptr;
    }

    return &text[length];
}

}    // namespace MGL::Text

// vim: et ts=4 sw=4 sts=4
