// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_text_format_argument.cc
 *  \brief      MGL テキストフォーマットの引数
 *  \date       Since: May 28, 2021. 22:42:09 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/text/mgl_text_format_argument.h>

#include <cmath>

#include <mgl/system/mgl_system_debug_macro.h>
#include <mgl/system/mgl_system_locale.h>

namespace MGL::Text
{
namespace
{
constexpr size_t kWidthMax = 127;
constexpr int kPrecisionMax = 7;
}    // namespace
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字列に変換
 *  \param[in]  options      変換オプション
 *  \return     変換後の文字列
 */
/* ------------------------------------------------------------------------- */
STL::string FormatArgument::ToString(const FormatOptions &options) const noexcept
{
    STL::string result;
    
    switch (_type)
    {
        // 値
        case Type::Int:
        case Type::UnsignedInt:
        case Type::Float:
            if ((options.valueDisplayMode == FormatOptions::ValueDisplayMode::UpperCaseHex) ||
                (options.valueDisplayMode == FormatOptions::ValueDisplayMode::LowerCaseHex))
            {
                return ToHexStringFromValue(options);
            }
            else
            {
                return ToStringFromNumber(options);
            }

        // 論理値
        case Type::Bool:
            if (_b)
            {
                AlignmentString(result, options, "TRUE");
            }
            else
            {
                AlignmentString(result, options, "FALSE");
            }
            return result;
            
        // 文字列
        case Type::String:
            AlignmentString(result, options, _string);
            return result;
            
        // ポインタ
        case Type::Pointer:
            return ToHexStringFromValue(options);

        // それ以外は不正
        case Type::Void:
        default:
            break;
    }

    MGL_WARNING("[MGL Text] Detected invalid format argument.");
    AlignmentString(result, options, "N/A");
    return result;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フォーマット文字列のパース
 *  \param[out] options      オプションの格納先
 *  \param[in]  parseText   パースする文字列
 *  \return     パース後の先頭を指すポインタ．パース失敗時はparseTextと同じ
 */
/* ------------------------------------------------------------------------- */
const char *FormatArgument::Parse(FormatOptions &options, const char *parseText) noexcept
{
    // 文字列を数値に変換して次の文字列の先頭を取得するラムダ式を定義
    auto getNumber = [](int &number, const char *text) noexcept
    {
        STL::string str;
        while ((*text >= '0') && (*text <= '9'))
        {
            str += *text;
            text++;
        }

        if (!str.empty())
        {
            number = std::stoi(std::string(str));
        }

        return text;
    };

    options = FormatOptions();
    const char *current = parseText;

    // 最初の文字が'{'でなければフォーマット文字列ではない
    if (*current != '{')
    {
        return parseText;
    }
    current++;

    // 2番目の文字列も'{'であればエスケープなのでフォーマット文字列でない
    if (*current == '{')
    {
        return current;
    }

    // 2番目の文字列が'}'であれば内容を全て省略しているのでデフォルトのオプションを返す
    if (*current == '}')
    {
        options.isValid = true;

        return ++current;
    }

    // 2番目の文字列が数値であればインデックス指定
    if ((*current >= '0') && (*current <= '9'))
    {
        current = getNumber(options.index, current);
        if (*current == '\0')
        {
            return parseText;
        }
    }

    // 以降はオプションの指定なので'}'が来るまで文字を解析する
    while (*current != '}')
    {
        bool skipIncrement = false;

        switch (*current)
        {
            // '}'で閉じていない場合はフォーマット文字列として扱わない
            case '\0':
                return parseText;

            // '+'は正の符号を表示
            case '+':
                options.showPlusSign = true;
                break;

            // ','は区切り文字を表示
            case ',':
                options.showSeparator = true;
                break;

            // '<'は左寄せ
            case '<':
                current = getNumber(options.width, ++current);
                options.width = -options.width;
                skipIncrement = true;
                break;

            // '>'は右寄せ
            case '>':
                current = getNumber(options.width, ++current);
                skipIncrement = true;
                break;

            // '.'は小数点以下の桁数の指定
            case '.':
                current = getNumber(options.precision, ++current);
                skipIncrement = true;
                break;

            // ’f’は右寄せまたは左寄せの際に埋める文字
            case 'f':
                current++;
                options.fillCharacter = *current;
                break;

            // '~'は自動インデックスをスキップ
            case '~':
                options.skipAutoIndex = true;
                break;

            // 'x'は小文字の16進数表示
            case 'x':
                current = getNumber(options.hexWidth, ++current);
                options.valueDisplayMode = FormatOptions::ValueDisplayMode::LowerCaseHex;
                skipIncrement = true;
                break;

            // 'X'は大文字の16進数表示
            case 'X':
                current = getNumber(options.hexWidth, ++current);
                options.valueDisplayMode = FormatOptions::ValueDisplayMode::UpperCaseHex;
                skipIncrement = true;
                break;

            // ':'と' 'は何もしない
            case ':':
            case ' ':
                break;

            // それ以外は不明なオプション（エラーにはせず警告ログだけ）
            default:
                MGL_WARNING("[MGL Text] Unknown format options: %c", *current);
                break;
        }

        if (!skipIncrement)
        {
            current++;
        }
    }

    options.isValid = true;

    return ++current;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      置換オプションをインデックス文字に変換
 *  \param[in]  options  オプション
 *  \return     置換オプションを表すインデックス文字
 */
/* ------------------------------------------------------------------------- */
STL::vector<IndexedCharacter> FormatArgument::ToIndexedCharacters(const FormatOptions &options) noexcept
{
    STL::vector<IndexedCharacter> indexes;
    indexes.reserve(0xF);
    const FormatOptions defaultOptions;

    // インデックス
    if (options.index != defaultOptions.index)
    {
        indexes.push_back(kSubstituteFormatMaskIndex | static_cast<uint8_t>(options.index));
    }

    // 桁幅
    if (options.width != defaultOptions.width)
    {
        indexes.push_back(kSubstituteFormatMaskWidth | static_cast<uint8_t>(options.width));
    }

    // 桁埋め文字
    if (options.fillCharacter != defaultOptions.fillCharacter)
    {
        indexes.push_back(kSubstituteFormatMaskFillCharacter | static_cast<uint8_t>(options.fillCharacter));
    }

    // 小数点以下の桁幅
    if (options.precision != defaultOptions.precision)
    {
        indexes.push_back(kSubstituteFormatMaskPrecision | static_cast<uint8_t>(options.precision));
    }

    // 16進数の桁幅
    if (options.hexWidth != defaultOptions.hexWidth)
    {
        indexes.push_back(kSubstituteFormatMaskHexWidth | static_cast<uint8_t>(options.hexWidth));
    }

    // 各種フラグ
    if ((options.showPlusSign != defaultOptions.showPlusSign) ||
        (options.showSeparator != defaultOptions.showSeparator) ||
        (options.valueDisplayMode != defaultOptions.valueDisplayMode) ||
        (options.skipAutoIndex != defaultOptions.skipAutoIndex))
    {
        uint8_t flags = 0;

        // 符号の表示
        if (options.showPlusSign)
        {
            flags |= 1u << kSubstituteFlagBitShowPlusSign;
        }
        // 区切り文字の表示
        if (options.showSeparator)
        {
            flags |= 1u << kSubstituteFlagBitShowSeparator;
        }
        // 自動インデックスのスキップ
        if (options.skipAutoIndex)
        {
            flags |= 1u << kSubstituteFlagBitSkipAutoIndex;
        }

        // 値の表示モード
        flags |= static_cast<uint8_t>(static_cast<uint8_t>(options.valueDisplayMode) << kSubstituteFlagBitValueDisplayModeLo);

        indexes.push_back(kSubstituteFormatMaskFlags | flags);
    }

    return indexes;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      整数にしてゼロか否かを取得
 *  \retval     true    ゼロ
 *  \retval     false   ゼロではない
 */
/* ------------------------------------------------------------------------- */
bool FormatArgument::IsZero() const noexcept
{
    switch (_type)
    {
        case Type::UnsignedInt:
        case Type::Int:
            return _ui == 0;

        case Type::Float:
            return static_cast<uint32_t>(std::fabs(_f)) == 0;

        case Type::Bool:
            return !_b;

        case Type::String:
            return _string.empty();

        case Type::Void:
        default:
            break;
    }

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      値が正の数かを取得
 *  \retval     true    正の数
 *  \retval     false   負の数
 */
/* ------------------------------------------------------------------------- */
bool FormatArgument::IsPositiveValue() const noexcept
{
    if (_type == Type::UnsignedInt)
    {
        return true;
    }

    if (_type == Type::Int)
    {
        return _i >= 0;
    }

    if (_type == Type::Float)
    {
        return _f >= 0.0f;
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      値が負の数かを取得
 *  \retval     true    負の数
 *  \retval     false   正の数
 */
/* ------------------------------------------------------------------------- */
bool FormatArgument::IsNegativeValue() const noexcept
{
    if (_type == Type::Int)
    {
        return _i < 0;
    }

    if (_type == Type::Float)
    {
        return _f < 0.0f;
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      数値を文字列に変換
 *  \param[in]  options      表示オプション
 *  \return     変換後の文字列
 */
/* ------------------------------------------------------------------------- */
STL::string FormatArgument::ToStringFromNumber(const FormatOptions &options) const noexcept
{
    // 値を取得
    uint64_t value = 0;
    if (_type == Type::UnsignedInt)
    {
        value = _ui;
    }
    else if (_type == Type::Int)
    {
        value = static_cast<uint64_t>((_i >= 0) ? _i : (_i * -1));
    }
    else if (_type == Type::Float)
    {
        value = static_cast<uint64_t>(std::fabs(_f));
    }

    // 区切りの桁数と記号を取得
    const auto &locale = System::Locale();
    const int32_t separateNumberOfDigit = locale.GetGroupingSize();
    const char separateSymbol = *(locale.GetGroupingSeparator());
    int32_t separateCount = separateNumberOfDigit;

    char buffer[kWidthMax + 1];
    buffer[kWidthMax] = '\0';
    auto *top = &buffer[kWidthMax - 1];
    while (buffer <= top)
    {
        // 値が0なら終了
        if (value == 0)
        {
            // そもそも元の値が0
            if (IsZero())
            {
                *top = '0';
            }
            // オプション指定がある場合，タイプが符号なし or 符号付きで正の数なら'+'を付ける
            else if (options.showPlusSign && IsPositiveValue())
            {
                *top = '+';
            }
            // 符号付きで負数なら'-'を付ける
            else if (IsNegativeValue())
            {
                *top = '-';
            }
            // それ以外
            else
            {
                top++;
            }

            break;
        }

        // オプション指定があり，区切り記号が必要な桁なら記号を付ける
        if (options.showSeparator && (separateCount <= 0))
        {
            *top = separateSymbol;
            separateCount = separateNumberOfDigit;
        }
        // それ以外は下1桁を文字に変換
        else
        {
            separateCount--;
            *top = static_cast<char>('0' + static_cast<char>((value % 10)));
            value /= 10;
        }

        top--;
    }

    STL::string numberString(top);

    // 小数点数の場合は小数部の部分も変換して追加
    if ((_type == Type::Float) && (options.precision != 0))
    {
        numberString += ToStringFromDecimal(options);
    }

    STL::string result;
    AlignmentString(result, options, numberString);

    return result;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          小数部の値を文字列に変換
 *  \param[in]      options      表示オプション
 */
/* ------------------------------------------------------------------------- */
STL::string FormatArgument::ToStringFromDecimal(const FormatOptions &options) const noexcept
{
    // 区切りの桁数と記号を取得
    const System::Locale locale;
    const int32_t separateNumberOfDigit = locale.GetGroupingSize();
    const char pointSymbol = *(locale.GetDecimalSeparator());
    int32_t separateCount = separateNumberOfDigit;

    STL::string result;

    // 小数点を追加
    result += pointSymbol;

    // 整数部を除いた値を取得
    float f = std::fabs(_f);
    f -= std::floor(f);

    // 必要な桁数分ループ
    const int max = (options.precision >= 0) ? options.precision : kPrecisionMax;
    for (int i = 0; i < max; i++)
    {
        // 小数部を10倍した値の整数部を文字に変換
        f *= 10.0f;
        const float n = std::floor(f);
        result += static_cast<char>('0' + static_cast<char>(n));
        f -= n;    // 整数部は消す

        // 桁数が指定されていない場合，値が一定以下になった時点でループを抜ける
        if ((options.precision < 0) && (f < 0.0000001f))
        {
            break;
        }

        // 区切り指定がある場合は一定桁数おきに区切り文字を挿入する
        if (options.showSeparator && (i <= (max - 2)))
        {
            if (--separateCount <= 0)
            {
                result += ' ';
                separateCount = separateNumberOfDigit;
            }
        }
    }

    return result;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      値を16進数の文字列に変換
 *  \param[in]  options  オプション
 *  \return     16進数の文字列
 */
/* ------------------------------------------------------------------------- */
STL::string FormatArgument::ToHexStringFromValue(const FormatOptions &options) const noexcept
{
    constexpr const char *kUpperCaseCharacterTable = "0123456789ABCDEF";    // 大文字用テーブル
    constexpr const char *kLowerCaseCharacterTable = "0123456789abcdef";    // 小文字用テーブル

    // テーブルの選択
    const auto *characterTable = kUpperCaseCharacterTable;
    if (options.valueDisplayMode == FormatOptions::ValueDisplayMode::LowerCaseHex)
    {
        characterTable = kLowerCaseCharacterTable;
    }

    STL::string reverseString;
    int count = 0;

    // 値を16進数の文字列の変換
    uint64_t value = _ui;
    while (value != 0)
    {
        auto v = (value & 0x0Fu);

        reverseString += characterTable[v];

        count++;

        value >>= 4u;
    }

    // reverseStringには逆順に格納されているので反転させる
    STL::string str;
    for (auto it = reverseString.rbegin(); it != reverseString.rend(); ++it)
    {
        str += *it;
    }

    // 指定した桁数だけ0を詰める
    if (options.hexWidth > count)
    {
        STL::string fill;
        auto fillCount = options.hexWidth - count;
        while (fillCount-- != 0)
        {
            fill += '0';
        }

        str = fill + str;
    }

    STL::string result;
    AlignmentString(result, options, str);
    return result;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字列のアライメント処理
 *  \param[out] result  処理結果
 *  \param[in]  options オプション
 *  \param[in]  string  変換元の文字列
 *  \note
 *      この関数はマルチバイト文字に対応していない
 */
/* ------------------------------------------------------------------------- */
void FormatArgument::AlignmentString(STL::string &result, const FormatOptions &options, const STL::string &string) noexcept
{
    result.clear();
    
    // アライメント指定が無ければ何もしない
    if (options.width == 0)
    {
        result = string;
        return;
    }
    auto width = static_cast<size_t>(std::abs(options.width));

    // NOTE:
    //  length()は要素数の取得なのでマルチバイト文字を正しく処理できない．
    //  正しく対応するなら文字単位の取得とその文字が全角であるかどうかの取得が必要．
    auto length = string.length();

    // 指定された幅より長い文字列に対しては何もしない
    if (length >= width)
    {
        result = string;
        return;
    }

    // 詰める文字列を生成
    STL::string fill;
    auto fillCount = width - length;
    while (fillCount-- != 0)
    {
        fill += options.fillCharacter;
    }

    // 右寄せの場合は詰める文字列を先に配置
    if (options.width > 0)
    {
        result = fill + string;
    }
    // 左寄せの場合は詰める文字列を後に配置
    else
    {
        result = string + fill;
    }
}

}    // namespace MGL::Text

// vim: et ts=4 sw=4 sts=4
