// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_text_format.cc
 *  \brief      MGL テキストフォーマット
 *  \date       Since: May 29, 2021. 15:24:35 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/text/mgl_text_format.h>

namespace MGL::Text
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字列のフォーマット
 *  \param[in]  text    変換元の文字列
 *  \param[in]  args    引数
 *  \return     変換後の文字列
 */
/* ------------------------------------------------------------------------- */
STL::string Format(const STL::string &text, const FormatArgs &args) noexcept
{
    STL::string dest;

    const char *source = text.c_str();
    FormatOptions options;
    size_t autoIndex = 0;

    while (*source != '\0')
    {
        // 現在の先頭からパースして有効であれば対応した引数に置き換える
        source = FormatArgument::Parse(options, source);
        if (options.isValid)
        {
            // インデックスの指定があればそのインデックス，無ければ自動インデックスを使う
            size_t argsIndex = autoIndex;
            if (options.index >= 0)
            {
                argsIndex = static_cast<size_t>(options.index);
            }

            // インデックスが引数の有効範囲内であれば置き換える
            if (argsIndex < args.size())
            {
                dest += args[argsIndex].ToString(options);
            }
            // 有効範囲外
            else
            {
                dest += "<!OOR!>";    // Out Of Range
            }

            // 自動インデックスをスキップするオプションが指定されていなければ更新
            if (!options.skipAutoIndex)
            {
                autoIndex++;
            }
        }
        // 置き換え対象でなければ出力先に文字を追加
        else
        {
            dest += *source;
            source++;
        }
    }

    return dest;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      インデックス文字列のフォーマット
 *  \param[in]  indexedText     変換元のインデックス文字列
 *  \param[in]  indexConverter  引数の文字列をインデックス化するためのコンバータ
 *  \param[in]  args            引数
 *  \return     変換後のインデックス文字列
 */
/* ------------------------------------------------------------------------- */
STL::vector<IndexedCharacter> Format(const IndexedCharacter *indexedText, const IndexConverter &indexConverter, const FormatArgs &args) noexcept
{
    STL::vector<IndexedCharacter> dest;

    FormatOptions options;
    options.isValid = true;
    size_t autoIndex = 0;

    for (size_t offset = 0; indexedText[offset] != kIndexedCharacterEndOfText; offset++)
    {
        auto index = indexedText[offset];

        // インデックスがフォーマットオプションであれば解析
        if ((index >= kIndexedCharacterSubstituteOptionReserveStart) && (index <= kIndexedCharacterSubstituteOptionReserveEnd))
        {
            const uint8_t value = index & 0xFFu;
            switch (auto substType = index & 0xFF00u; substType)
            {
                // インデックス
                case kSubstituteFormatMaskIndex:
                    options.index = value;
                    break;

                // 桁幅
                case kSubstituteFormatMaskWidth:
                    options.width = static_cast<int>(value);
                    break;

                // 小数点以下の桁幅
                case kSubstituteFormatMaskPrecision:
                    options.precision = static_cast<int>(value);
                    break;

                // 桁幅埋め文字
                case kSubstituteFormatMaskFillCharacter:
                    options.fillCharacter = static_cast<char>(value);
                    break;

                // 16進数の桁幅
                case kSubstituteFormatMaskHexWidth:
                    options.hexWidth = value;
                    break;

                // 各種フラグ
                case kSubstituteFormatMaskFlags:
                    options.showPlusSign = (value & (1u << kSubstituteFlagBitShowPlusSign)) != 0;
                    options.showSeparator = (value & (1u << kSubstituteFlagBitShowSeparator)) != 0;
                    options.skipAutoIndex = (value & (1u << kSubstituteFlagBitSkipAutoIndex)) != 0;
                    options.valueDisplayMode = static_cast<FormatOptions::ValueDisplayMode>(
                        static_cast<uint8_t>((value >> kSubstituteFlagBitValueDisplayModeLo)) & 0x03u);
                    break;

                // それ以外は無視
                default:
                    break;
            }
        }
        // インデックスが置換であれば引数で置き換え
        else if (index == kIndexedCharacterSubstitute)
        {
            // インデックスの指定があればそのインデックス，無ければ自動インデックスを使う
            size_t argsIndex = autoIndex;
            if (options.index >= 0)
            {
                argsIndex = static_cast<size_t>(options.index);
            }

            STL::string argumentString;

            // インデックスが引数の有効範囲内であれば置き換え文字を取得
            if (argsIndex < args.size())
            {
                argumentString += args[argsIndex].ToString(options);
            }
            // 有効範囲外
            else
            {
                argumentString += "<!OOR!>";    // Out Of Range
            }

            // 置き換え文字をインデックス化して追加
            if (auto indexedArgumentString = indexConverter.ToIndexedString(argumentString.c_str(), false, false); indexedArgumentString != nullptr)
            {
                for (size_t i = 0; indexedArgumentString[i] != kIndexedCharacterEndOfText; i++)
                {
                    dest.push_back(indexedArgumentString[i]);
                }
            }

            // 自動インデックスをスキップするオプションが指定されていなければ更新
            if (!options.skipAutoIndex)
            {
                autoIndex++;
            }

            // オプションは一度使用したらデフォルトに戻す
            options = FormatOptions();
            options.isValid = true;
        }
        // それ以外は追加
        else
        {
            dest.push_back(index);
        }
    }

    // 最後に終端文字を書き込んでおく
    dest.push_back(MGL::Text::kIndexedCharacterEndOfText);

    return dest;
}

}    // namespace MGL::Text

// vim: et ts=4 sw=4 sts=4
