// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_text_converter.cc
 *  \brief      MGL テキストコンバータ
 *  \date       Since: June 29, 2022. 1:15:11 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/text/mgl_text_converter.h>

#include <mgl/text/mgl_text_stream.h>

namespace MGL::Text
{
namespace
{
constexpr size_t kReserveSize = 1024 - 64;
}
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      UTF-8文字列に変換
 *  \param[out] dest        出力先
 *  \param[in]  src         変換元のバッファ
 *  \param[in]  size        変換元のバッファのサイズ
 *  \param[in]  encoding    変換元のエンコードタイプ
 *  \param[in]  existBOM    変換元にBOMが存在しているかのフラグ
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool ToUTF8(STL::string &dest, const void *src, size_t size, Encoding encoding, bool existBOM) noexcept
{
    dest.clear();
    dest.reserve(kReserveSize);

    TextStream stream(reinterpret_cast<const uint8_t *>(src), size, encoding, existBOM);

    if (stream.GetEncoding() == Encoding::UTF8)
    {
        return false;
    }

    while (!stream.IsTerminated())
    {
        auto character = stream.Read();
        if (!character.IsValid())
        {
            dest.clear();
            return false;
        }

        if (character.IsNull())
        {
            break;
        }

        dest += character.ToString();
    }

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      UTF-8文字列に変換
 *  \param[out] dest        出力先
 *  \param[in]  destSize    出力先バッファのサイズ
 *  \param[in]  src         変換元のバッファ
 *  \param[in]  srcSize     変換元のバッファのサイズ
 *  \param[in]  encoding    変換元のエンコードタイプ
 *  \param[in]  existBOM    変換元にBOMが存在しているかのフラグ
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool ToUTF8(char *dest, size_t destSize, const void *src, size_t srcSize, Encoding encoding, bool existBOM) noexcept
{
    TextStream stream(reinterpret_cast<const uint8_t *>(src), srcSize, encoding, existBOM);

    if (stream.GetEncoding() == Encoding::UTF8)
    {
        return false;
    }

    auto remainSize = destSize;

    while (!stream.IsTerminated())
    {
        auto character = stream.Read();
        if (!character.IsValid())
        {
            return false;
        }

        dest = character.ToByte(dest, remainSize);
        if (dest == nullptr)
        {
            return false;
        }

        if (character.IsNull())
        {
            break;
        }

        remainSize -= character.GetByteSize();
    }

    return true;
}

}    // namespace MGL::Text

// vim: et ts=4 sw=4 sts=4
