// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_text_character.cc
 *  \brief      文字クラス
 *  \date       Since: October 4, 2018. 2:20:42 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/text/mgl_text_character.h>

#include <cstring>

namespace MGL::Text
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字をバイトに変換する際に必要なサイズを取得
 *  \return     必要サイズ．不正な文字の場合は0
 */
/* ------------------------------------------------------------------------- */
size_t Character::GetByteSize() const noexcept
{
    if (IsValid())
    {
        // 1Byte
        if (_code <= 0x7F)
        {
            return 1;
        }

        // 2Byte
        if (_code <= 0x7FF)
        {
            return 2;
        }

        // 3Byte
        if (_code <= 0xFFFF)
        {
            return 3;
        }

        // 4Byte
        if (_code <= 0x1FFFFF)
        {
            return 4;
        }
    }

    return 0;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字をバイト列に変換
 *  \param[out] dest        書き込み先のアドレス
 *  \param[in]  destSize    書き込み先バッファのサイズ
 *  \return     成功した場合は終端記号を書き込んだアドレス．失敗時にはnullptr
 */
/* ------------------------------------------------------------------------- */
char *Character::ToByte(char *dest, size_t destSize) const noexcept
{
    // サイズを取得して0なら失敗
    auto byteSize = GetByteSize();
    if (byteSize == 0)
    {
        return nullptr;
    }

    // 書き込み先に必要なサイズが無ければ失敗
    if (destSize <= (byteSize + 1))
    {
        return nullptr;
    }

    // サイズに応じてデコード
    switch (byteSize)
    {
        // 1Byte
        case 1:
        {
            *(dest++) = static_cast<char>(_code);
            break;
        }

        // 2Byte
        case 2:
        {
            const uint32_t code = _code & 0x7FFu;
            *(dest++) = static_cast<char>(0xC0u | (code >> 6u));
            *(dest++) = static_cast<char>(0x80u | (code & 0x3Fu));
            break;
        }

        // 3Byte
        case 3:
        {
            const uint32_t code = _code & 0xFFFFu;
            *(dest++) = static_cast<char>(0xE0u | (code >> 12u));
            *(dest++) = static_cast<char>(0x80u | ((code >> 6u) & 0x3Fu));
            *(dest++) = static_cast<char>(0x80u | (code & 0x3Fu));
            break;
        }

        // 4Byte
        case 4:
        {
            const uint32_t code = _code & 0x1FFFFFu;
            *(dest++) = static_cast<char>(0xF0u | (code >> 18u));
            *(dest++) = static_cast<char>(0x80u | ((code >> 12u) & 0x3Fu));
            *(dest++) = static_cast<char>(0x80u | ((code >> 6u) & 0x3Fu));
            *(dest++) = static_cast<char>(0x80u | (code & 0x3Fu));
            break;
        }

        // それ以外はnullptr
        default:
            return nullptr;
    }

    // 終端コードを書き込んでそのアドレスを返す
    *dest = '\0';
    return dest;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コードをバイト列に変換
 *  \param[out] byteArray   バイトコードの格納先
 *  \retval     true        成功
 *  \retval     false       失敗
 *  \note
 *      とりあえずUTF-8で返す．
 *      将来は他の他の文字コード（Shift-JISなど）にも対応するかも．
 */
/* ------------------------------------------------------------------------- */
bool Character::ToByte(char byteArray[kByteArraySize]) const noexcept
{
    if (!IsValid())
    {
        return false;
    }

    memset(byteArray, 0, kByteArraySize);

    // 1Byte
    if (_code <= 0x7F)
    {
        byteArray[0] = static_cast<char>(_code);
    }
    // 2Byte
    else if (_code <= 0x7FF)
    {
        const uint32_t code = _code & 0x7FFu;
        byteArray[0] = static_cast<char>(0xC0u | (code >> 6u));
        byteArray[1] = static_cast<char>(0x80u | (code & 0x3Fu));
    }
    // 3Byte
    else if (_code <= 0xFFFF)
    {
        const uint32_t code = _code & 0xFFFFu;
        byteArray[0] = static_cast<char>(0xE0u | (code >> 12u));
        byteArray[1] = static_cast<char>(0x80u | ((code >> 6u) & 0x3Fu));
        byteArray[2] = static_cast<char>(0x80u | (code & 0x3Fu));
    }
    // 4Byte
    else if (_code <= 0x1FFFFF)
    {
        const uint32_t code = _code & 0x1FFFFFu;
        byteArray[0] = static_cast<char>(0xF0u | (code >> 18u));
        byteArray[1] = static_cast<char>(0x80u | ((code >> 12u) & 0x3Fu));
        byteArray[2] = static_cast<char>(0x80u | ((code >> 6u) & 0x3Fu));
        byteArray[3] = static_cast<char>(0x80u | (code & 0x3Fu));
    }
    // それ以降はUTF-8では不正なシーケンスなので失敗
    else
    {
        return false;
    }

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字をSTL::stringに変換
 *  \return     変換した文字
 */
/* ------------------------------------------------------------------------- */
STL::string Character::ToString() const noexcept
{
    char byteArray[kByteArraySize];
    if (ToByte(byteArray))
    {
        return {byteArray};
    }

    return {""};
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字列をSTL::stringに変換
 *  \return     変換した文字列
 */
/* ------------------------------------------------------------------------- */
STL::string CharacterArray::ToString() const
{
    STL::string str;

    for (const auto &character : *this)
    {
        str += character.ToString();
    }

    return str;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字列の一部をSTL::stringに変換
 *  \param[in]  start   開始位置
 *  \param[in]  end     終了位置
 *  \return     startからendまでの文字を変換した文字列
 */
/* ------------------------------------------------------------------------- */
STL::string CharacterArray::ToString(uint32_t start, uint32_t end) const
{
    auto arraySize = static_cast<uint32_t>(size());
    if ((end == 0) || (end >= arraySize))
    {
        end = arraySize - 1;
    }

    STL::string str;

    for (uint32_t i = start; i < end; i++)
    {
        str += at(i).ToString();
    }

    return str;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字検索
 *  \param[in]  findCode    検索する文字（Unicodeスカラ値）
 *  \param[in]  start       検索開始位置
 *  \return     見つかった場合はそのインデックス，見つからない場合は負数．
 */
/* ------------------------------------------------------------------------- */
int CharacterArray::Find(uint32_t findCode, uint32_t start) const
{
    auto arraySize = static_cast<uint32_t>(size());

    for (uint32_t i = start; i < arraySize; i++)
    {
        if (at(i).GetCode() == findCode)
        {
            return static_cast<int>(i);
        }
    }

    return -1;
}
}    // namespace MGL::Text

// vim: et ts=4 sw=4 sts=4

