// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_task_weak_node.cc
 *  \brief      MGL 弱参照タスクノード
 *  \date       Since: June 3, 2021. 10:48:10 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/task/mgl_task_weak_node.h>

#include <mgl/task/mgl_task_node_list.h>

namespace MGL::Task
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  listIndex   ノードリスト上のインデックス
 */
/* ------------------------------------------------------------------------- */
WeakNode::WeakNode(size_t listIndex) noexcept
    : WeakNode()
{
    Set(listIndex);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  node   タスクノード
 */
/* ------------------------------------------------------------------------- */
WeakNode::WeakNode(const Node *node) noexcept
    : WeakNode()
{
    Set(node);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ノードの設定
 *  \param[in]  listIndex   ノードリスト上のインデックス
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool WeakNode::Set(size_t listIndex) noexcept
{
    const auto *nodeElement = NodeList::GetInstance().GetElement(listIndex);
    if (nodeElement != nullptr)
    {
        if (nodeElement->uniqueID != UniqueIdentifier::Invalid)
        {
            _listIndex = listIndex;
            _identifier = nodeElement->node->GetIdentifier();
            _uniqueID = nodeElement->uniqueID;

            return true;
        }
    }

    _listIndex = 0;
    _identifier = Identifier{0};
    _uniqueID = UniqueIdentifier::Invalid;

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ノードの設定
 *  \param[in]  node    タスクノード
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool WeakNode::Set(const Node *node) noexcept
{
    const auto *nodeElement = NodeList::GetInstance().GetElement(node);
    if (nodeElement != nullptr)
    {
        if (nodeElement->uniqueID != UniqueIdentifier::Invalid)
        {
            _listIndex = nodeElement->arrayIndex;
            _identifier = nodeElement->node->GetIdentifier();
            _uniqueID = nodeElement->uniqueID;

            return true;
        }
    }

    _listIndex = 0;
    _identifier = Identifier{0};
    _uniqueID = UniqueIdentifier::Invalid;

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ノードの取得
 *  \return     参照先のタスクノード．既に寿命が切れている場合はnullptr
 */
/* ------------------------------------------------------------------------- */
Node *WeakNode::Get() const noexcept
{
    if (_uniqueID == UniqueIdentifier::Invalid)
    {
        return nullptr;
    }

    const auto *nodeElement = NodeList::GetInstance().GetElement(_listIndex);
    if (nodeElement == nullptr)
    {
        return nullptr;
    }

    if (nodeElement->uniqueID == _uniqueID)
    {
        if (nodeElement->node->GetIdentifier() == _identifier)
        {
            return nodeElement->node.get();
        }
    }

    return nullptr;
}
};    // namespace MGL::Task

// vim: et ts=4 sw=4 sts=4
