// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_task_thread_pool.cc
 *  \brief      MGL タスクシステム用スレッドプール
 *  \date       Since: November 3, 2023. 22:14:20 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/task/mgl_task_thread_pool.h>

#include <mgl/task/mgl_task_node.h>

namespace MGL::Task
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \param[in]  threadCount     生成するスレッド数
 *  \return     生成されたスレッド数
 */
/* ------------------------------------------------------------------------- */
int32_t ThreadPool::Initialize(int32_t threadCount) noexcept
{
    // 既に初期化済みなら何もせず現在のスレッド数を返す
    if (_isAvailable)
    {
        return static_cast<int32_t>(_threads.size());
    }

    // スレッド数が1未満であれば失敗（並列化する意味がないため）
    if (threadCount <= 1)
    {
        return 0;
    }

    // 要求された数だけスレッドを生成
    for (int32_t i = 0; i < threadCount; i++)
    {
        Element element;

        element.thread = STL::make_unique<Thread>(*this, static_cast<uint32_t>(i));

        if (!element.thread->Create())
        {
            _threads.clear();
            break;
        }

        _threads.push_back(std::move(element));
    }

    // 生成したスレッドから空きリストを構築
    const auto createCount = static_cast<int32_t>(_threads.size());
    if (createCount != 0)
    {
        _freeTop.next = 0;
        for (size_t i = 0; i < _threads.size() - 1; i++)
        {
            _threads[i].next = static_cast<int32_t>(i + 1);
        }

        _isAvailable = true;
    }

    return createCount;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      スレッドを用いてタスクを実行
 *  \param[in]  node    実行するタスクノード
 *  \param[in]  stage   実行するタスクのステージ
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool ThreadPool::Execute(Node *node, ExecuteStage stage) noexcept
{
    // スレッドプールが有効でなければ失敗
    if (!_isAvailable)
    {
        return false;
    }

    // 実行可能になるまで待つ
    WaitForExecutable();

    const std::scoped_lock lock(_mutex);

    // 空きスレッドのインデックスが0未満であれば失敗
    // （通常ここは通らないが念の為）
    if (_freeTop.next < 0)
    {
        return false;
    }

    // 空きスレッドの情報を更新
    const auto &element = _threads[static_cast<size_t>(_freeTop.next)];
    _freeTop.next = element.next;

    // 実行カウンタを更新
    {
        const std::unique_lock conditionLock(_conditionMutex);
        _executeCount++;

        if (node->IsEnabledBarrier())
        {
            _barrierMap[node->GetBarrierIdentifier()]++;
        }
    }

    // スレッドを実行
    element.thread->Execute(node, stage);

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      実行可能になるまで待機
 */
/* ------------------------------------------------------------------------- */
void ThreadPool::WaitForExecutable() noexcept
{
    std::unique_lock lock(_conditionMutex);

    _condition.wait(lock, [this]
    {
        return _freeTop.next >= 0;
    });
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      全てのタスクを終えるまで待機
 */
/* ------------------------------------------------------------------------- */
void ThreadPool::WaitForComplete() noexcept
{
    std::unique_lock lock(_conditionMutex);

    _condition.wait(lock, [this]
    {
        return _executeCount == 0;
    });
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      バリア同期を行う
 *  \param[in]  identifier  同期するID
 */
/* ------------------------------------------------------------------------- */
void ThreadPool::SyncBarrier(Identifier identifier) noexcept
{
    std::unique_lock lock(_conditionMutex);

    _condition.wait(lock, [this, identifier]
    {
        return _barrierMap[identifier] == 0;
    });
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      完了通知
 *  \param[in]  index   完了したスレッドのインデックス
 *  \param[in]  node    完了したタスクノード
 */
/* ------------------------------------------------------------------------- */
void ThreadPool::OnCompleted(uint32_t index, const Node *node) noexcept
{
    const std::scoped_lock lock(_mutex);
    const std::unique_lock uniqueLock(_conditionMutex);

    auto &element = _threads[index];
    element.next = _freeTop.next;
    _freeTop.next = static_cast<int32_t>(index);
    _executeCount--;
    if (node->IsEnabledBarrier())
    {
        _barrierMap[node->GetBarrierIdentifier()]--;
    }

    _condition.notify_all();
}
}    // namespace MGL::Task

// vim: et ts=4 sw=4 sts=4
