// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_task_thread.cc
 *  \brief      MGL タスクシステム用スレッド
 *  \date       Since: November 3, 2023. 22:54:26 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/task/mgl_task_thread.h>

#include <system_error>

#include <mgl/task/mgl_task_node.h>

namespace MGL::Task
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
Thread::Thread(ThreadListener &listener, uint32_t index) noexcept
    : _listener(listener)
    , _index(index)
{
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デストラクタ
 */
/* ------------------------------------------------------------------------- */
Thread::~Thread() noexcept
{
    // 終了要求
    {
        const std::scoped_lock lock(_mutex);
        _shouldExit = true;
        _condition.notify_one();
    }

    // 終了するまで待つ
    _thread.join();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      スレッドの生成
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool Thread::Create() noexcept
{
    try
    {
        _thread = std::thread([this]
        {
            Process();
        });
    }
    catch ([[maybe_unused]] std::system_error &error)
    {
        return false;
    }

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タスクノードの実行
 *  \param[in]  taskNode    実行するタスクノード
 *  \param[in]  stage       実行ステージ
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool Thread::Execute(Node *taskNode, ExecuteStage stage) noexcept
{
    // 実行中や引数がnullptrの場合は失敗
    if (_isExecuting || (taskNode == nullptr))
    {
        return false;
    }

    const std::scoped_lock lock(_mutex);

    // 実行に必要なパラメータを設定
    _taskNode = taskNode;
    _stage = stage;

    // 実行要求
    _isExecuting = true;
    _condition.notify_one();

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      実行プロセス
 */
/* ------------------------------------------------------------------------- */
void Thread::Process() noexcept
{
    while (true)
    {
        // 通知が来るまで待機
        {
            std::unique_lock<std::mutex> lock(_mutex);

            _condition.wait(lock, [this]
            {
                return _isExecuting || _shouldExit;
            });
        }

        // 終了要求があればループを抜ける
        if (_shouldExit)
        {
            break;
        }
        // タスクノードを実行
        else
        {
            const auto *node = _taskNode;
            if (_taskNode != nullptr)
            {
                _taskNode->OnExecute(_stage);
                _taskNode = nullptr;
            }
            _isExecuting = false;

            // 実行が完了したことをリスナーに伝える
            _listener.OnCompleted(_index, node);
        }
    }
}
}    // namespace MGL::Task

// vim: et ts=4 sw=4 sts=4
