/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_task_node_sublist.cc
 *  \brief      MGL タスクリストのサブリスト
 *  \date       Since: March 30, 2023. 17:35:51 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/task/mgl_task_node_sublist.h>

namespace MGL::Task
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ノードリストの要素を追加
 *  \param[in]  element     追加する要素
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool NodeSubList::AddNodeListElement(NodeListElement *element) noexcept
{
    const std::scoped_lock lock(_mutex);

    MGL_ASSERT(element->node->GetIdentifier() == _identifier, "[Task] Detect addition of invalid node.");

    if (_standbyTail == nullptr)
    {
        _standbyTop = _standbyTail = element;
        element->next = element->previous = nullptr;
    }
    else
    {
        _standbyTail->next = element;
        element->next = nullptr;
        element->previous = _standbyTail->previous;
        _standbyTail = element;
    }

    _count++;

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      スタンバイ状態のタスクをアクティベート
 */
/* ------------------------------------------------------------------------- */
void NodeSubList::ActivateStandbyTask()
{
    // スタンバイ状態のタスクがなければ必要なし
    if (_standbyTop == nullptr)
    {
        return;
    }

    const std::scoped_lock lock(_mutex);

    // アクティブリストの末尾にスタンバイリストを接続
    if (_activeTail == nullptr)
    {
        _activeTop = _activeTail = _standbyTop;
    }
    else
    {
        _activeTail->next = _standbyTop;
        _standbyTop->previous = _activeTail;
        _activeTail = _standbyTail;
    }

    // スタンバイリストの初期化
    _standbyTop = _standbyTail = nullptr;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タスクリストの実行と削除
 *  \param[in]  stage   実行ステージ
 *  \param[in]  freeTop フリーリストの先頭
 */
/* ------------------------------------------------------------------------- */
void NodeSubList::ExecuteAndRemove(ExecuteStage stage, NodeListElement *freeTop) noexcept
{
    auto *current = _activeTop;
    while (current != nullptr)
    {
        auto *next = current->next;

        // 削除要求がなければ実行
        if (!current->node->IsRequestedKill())
        {
            current->node->OnExecute(stage);
        }

        // 削除要求があれば削除
        if (current->node->IsRequestedKill())
        {
            Remove(current, freeTop);
        }

        current = next;
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief          タスクリストを並列実行
 *  \param[in]      stage       実行ステージ
 *  \param[in,out]  threadPool  スレッドプール
 */
/* ------------------------------------------------------------------------- */
void NodeSubList::ParallelExecute(ExecuteStage stage, ThreadPool &threadPool) noexcept
{
    // バリア同期
    threadPool.SyncBarrier(_identifier);

    auto *current = _activeTop;
    while (current != nullptr)
    {
        auto *next = current->next;

        // 削除要求がなければ実行
        if (!current->node->IsRequestedKill())
        {
            // 非同期実行
            if (current->node->IsEnabledAsynchronous())
            {
                threadPool.Execute(current->node.get(), stage);
            }
            // 同期実行
            else
            {
                current->node->OnExecute(stage);
            }
        }

        current = next;
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      削除要求のあるノードを削除
 *  \param[in]  freeTop フリーリストの先頭
 */
/* ------------------------------------------------------------------------- */
void NodeSubList::Remove(NodeListElement *freeTop) noexcept
{
    auto *current = _activeTop;
    while (current != nullptr)
    {
        auto *next = current->next;

        // 削除要求があれば削除
        if (current->node->IsRequestedKill())
        {
            Remove(current, freeTop);
        }

        current = next;
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      削除要求
 *  \param[in]  resideLevel     常駐レベル
 *  \return     削除要求を発行した数
 */
/* ------------------------------------------------------------------------- */
uint32_t NodeSubList::Kill(ResideLevel resideLevel) noexcept
{
    uint32_t count = 0;

    ForEach([&](NodeListElement &element)
    {
        if (element.node->GetResideLevel() <= resideLevel)
        {
            element.node->Kill();
            count++;
        }

        return true;
    });

    return count;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ノードのリスト上の要素を取得
 *  \param[in]  node    取得するノード
 *  \return     リスト上のノードの要素
 */
/* ------------------------------------------------------------------------- */
const NodeListElement *NodeSubList::GetElement(const Node *node) noexcept
{
    const NodeListElement *foundElement = nullptr;

    ForEach([&](NodeListElement &element)
    {
        if (element.node.get() == node)
        {
            foundElement = &element;
            return false;
        }

        return true;
    });

    return foundElement;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ノードの配列を取得
 *  \param[out] weakNodeArray   弱参照ノードの配列
 */
/* ------------------------------------------------------------------------- */
void NodeSubList::GetNodeArray(STL::vector<WeakNode> &weakNodeArray) noexcept
{
    ForEach([&](NodeListElement &element)
    {
        weakNodeArray.emplace_back(element.node.get());
        return true;
    });
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タスクにイベントを通知
 *  \param[in]  eventIdentifier     イベントID
 *  \param[in]  argument            引数
 */
/* ------------------------------------------------------------------------- */
void NodeSubList::NotifyEvent(EventIdentifier eventIdentifier, void *argument) noexcept
{
    ForEach([&](NodeListElement &element)
    {
        element.node->NotifyEvent(eventIdentifier, argument);
        return true;
    });
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タスクの削除
 *  \param[in]  element     削除する要素
 *  \param[in]  freeTop     フリーリストの先頭
 */
/* ------------------------------------------------------------------------- */
void NodeSubList::Remove(NodeListElement *element, NodeListElement *freeTop) noexcept
{
    element->node.reset();
    element->uniqueID = UniqueIdentifier::Invalid;

    // タスクが最後の1つであればアクティブタスクの先頭と末尾をnullptrにする
    if ((element->next == nullptr) && (element->previous == nullptr))
    {
        _activeTop = _activeTail = nullptr;
    }
    // 最後の1つでない場合は削除対象の前後の要素をつなげる
    else
    {
        if (element->previous != nullptr)
        {
            element->previous->next = element->next;
        }
        else
        {
            _activeTop = element->next;
        }

        if (element->next != nullptr)
        {
            element->next->previous = element->previous;
        }
        else
        {
            _activeTail = element->previous;
        }
    }

    // 削除対象をフリーリストへ
    element->next = freeTop->next;
    freeTop->next = element;

    _count--;
}
}    // namespace MGL::Task

// vim: et ts=4 sw=4 sts=4
