// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_task_node_list.cc
 *  \brief      MGL タスクノードのリスト
 *  \date       Since: June 1, 2021. 17:06:05 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/task/mgl_task_node_list.h>

#include <mgl/system/mgl_system_debug_macro.h>

namespace MGL::Task
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      インスタンスの取得
 *  \return     インスタンスの参照
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<NodeList> &NodeList::GetInstanceRef() noexcept
{
    static STL::unique_ptr<NodeList> sInstance = nullptr;
    return sInstance;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \param[in]  capacity    タスクの最大数
 *  \param[in]  descriptor  初期化用記述子
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool NodeList::Initialize(size_t capacity, const InitializeDescriptor &descriptor) noexcept
{
    // 引数チェック
    if ((capacity == 0) || (descriptor.stageSettings.empty()))
    {
        return false;
    }

    // 各メンバを初期化
    _capacity = capacity;
    _nodeElements = STL::make_unique<NodeListElement[]>(capacity);
    _stageSettingsArray = descriptor.stageSettings;

    // リストを初期化
    _freeTopElement.next = &_nodeElements[0];
    for (size_t i = 0; i < _capacity; i++)
    {
        if (i < (_capacity - 1))
        {
            _nodeElements[i].next = &_nodeElements[i + 1];
        }

        if (i == 0)
        {
            _nodeElements[i].previous = &_freeTopElement;
        }
        else
        {
            _nodeElements[i].previous = &_nodeElements[i - 1];
        }

        _nodeElements[i].arrayIndex = i;
    }

    // 実行設定の指定が無ければステージ0の設定を1つだけ追加
    if (_stageSettingsArray.empty())
    {
        _stageSettingsArray.emplace_back(ExecuteStage{0}, ExecuteMode::NormalUpdate);
    }

    // スレッドプールの準備
    _threadPool.Initialize(descriptor.parallelExecuteCount);

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ノードの追加
 *  \param[in]  addNode     追加するノード
 *  \return     追加に成功した場合はリスト上のインデックス．失敗した場合は範囲外のインデックス
 */
/* ------------------------------------------------------------------------- */
size_t NodeList::Add(STL::unique_ptr<Node> addNode) noexcept
{
    const std::scoped_lock lock(_mutex);

    // 空きノードが無ければ失敗
    if (_freeTopElement.next == nullptr)
    {
        return _capacity;
    }

    // 追加するノードの要素を準備
    auto *addElement = _freeTopElement.next;
    addElement->node = std::move(addNode);
    _freeTopElement.next = addElement->next;

    // 対応するサブリストを取得
    auto *subList = GetSubList(addElement->node->GetIdentifier());
    if (subList == nullptr)
    {
        subList = CreateSubList(addElement->node->GetIdentifier());
        if (subList == nullptr)
        {
            return _capacity;
        }
    }

    // サブリストにノードの要素を追加
    subList->AddNodeListElement(addElement);

    // ユニークIDの設定
    addElement->uniqueID = _currentUniqueId++;

    // 初期化処理の呼び出し
    addElement->node->OnInitialize();

    return addElement->arrayIndex;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      実行
 */
/* ------------------------------------------------------------------------- */
void NodeList::Execute() noexcept
{
    // 全てのノードのスタンバイ状態を解除
    for (auto &subList : _subLists)
    {
        subList->ActivateStandbyTask();
    }

    // ステージの数だけノードのOnExecute()を呼び出す
    for (const auto &stageSettings : _stageSettingsArray)
    {
        // スレッドプールが無効な場合は並列実行を通常実行に変える
        auto executeMode = stageSettings.mode;
        if (executeMode == ExecuteMode::ParallelizableUpdate)
        {
            if (!_threadPool.IsAvailable())
            {
                executeMode = ExecuteMode::NormalUpdate;
            }
        }

        switch (executeMode)
        {
            // 通常の更新処理
            case ExecuteMode::NormalUpdate:
                for (auto &subList : _subLists)
                {
                    subList->ExecuteAndRemove(stageSettings.stage, &_freeTopElement);
                }
                break;

            // 並列化可能な更新処理
            case ExecuteMode::ParallelizableUpdate:
                // 並列実行
                for (auto &subList : _subLists)
                {
                    subList->ParallelExecute(stageSettings.stage, _threadPool);
                }

                // 全てのタスクが完了するまで待機
                _threadPool.WaitForComplete();

                // 削除要求のあるタスクを削除
                for (auto &subList : _subLists)
                {
                    subList->Remove(&_freeTopElement);
                }
                break;

            // 描画用の更新処理
            case ExecuteMode::RenderUpdate:
                // NOTE:
                //  現時点では通常の更新処理と同じだけど，ここは将来変更するかもしれない．
                for (auto &subList : _subLists)
                {
                    subList->ExecuteAndRemove(stageSettings.stage, &_freeTopElement);
                }
                break;
        }
    }

    // 空になったサブリストを削除
    for (auto it = _subLists.begin(); it != _subLists.end();)
    {
        if (it->get()->GetCount() == 0)
        {
            it = _subLists.erase(it);
        }
        else
        {
            ++it;
        }
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      全てのノードに削除要求を発行する
 *  \param[in]  resideLevel     常駐レベルがこの値以下のタスクを削除する
 *  \return     削除要求を発行したノードの数
 */
/* ------------------------------------------------------------------------- */
size_t NodeList::Kill(ResideLevel resideLevel) noexcept
{
    size_t count = 0;

    for (auto &subList : _subLists)
    {
        count += subList->Kill(resideLevel);
    }

    return count;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      指定したIDのノードに削除要求を発行する
 *  \param[in]  identifier      削除するID
 *  \param[in]  resideLevel     常駐レベルがこの値以下のタスクを削除する
 *  \return     削除要求を発行したノードの数
 */
/* ------------------------------------------------------------------------- */
size_t NodeList::Kill(Identifier identifier, ResideLevel resideLevel) noexcept
{
    if (auto *subList = GetSubList(identifier); subList != nullptr)
    {
        return subList->Kill(resideLevel);
    }

    return 0;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      全体のノード数を取得
 *  \return     全体のノード数
 */
/* ------------------------------------------------------------------------- */
size_t NodeList::GetCount() const noexcept
{
    size_t count = 0;

    for (const auto &subList : _subLists)
    {
        count += subList->GetCount();
    }

    return count;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      指定したIDのノード数を取得
 *  \return     指定したIDのノード数
 */
/* ------------------------------------------------------------------------- */
size_t NodeList::GetCount(Identifier identifier) const noexcept
{
    if (auto *subList = GetSubList(identifier); subList != nullptr)
    {
        return subList->GetCount();
    }

    return 0;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      リストの要素を取得
 *  \param[in]  index   要素のインデックス
 *  \return     インデックスに対応した要素．インデックスが範囲外の場合はnullptr
 */
/* ------------------------------------------------------------------------- */
const NodeListElement *NodeList::GetElement(size_t index) const noexcept
{
    if (index >= _capacity)
    {
        return nullptr;
    }

    return &_nodeElements[index];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      リストの要素を取得
 *  \param[in]  node    タスクノード
 *  \return     タスクノードに対応した要素．見つからない場合はnullptr
 */
/* ------------------------------------------------------------------------- */
const NodeListElement *NodeList::GetElement(const Node *node) noexcept
{
    auto *subList = GetSubList(node->GetIdentifier());
    if (subList == nullptr)
    {
        return nullptr;
    }

    return subList->GetElement(node);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ノードの配列を取得
 *  \param[out] weakNodeArray   弱参照ノードの配列
 *  \param[in]  identifier      取得するタスクのID
 */
/* ------------------------------------------------------------------------- */
void NodeList::GetNodeArray(STL::vector<WeakNode> &weakNodeArray, Identifier identifier) noexcept
{
    auto *subList = GetSubList(identifier);
    if (subList == nullptr)
    {
        weakNodeArray.clear();
    }
    else
    {
        subList->GetNodeArray(weakNodeArray);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タスクへのイベント通知
 *  \param[in]  eventIdentifier     イベントID
 *  \param[in]  argument            引数
 */
/* ------------------------------------------------------------------------- */
void NodeList::NotifyEvent(EventIdentifier eventIdentifier, void *argument) noexcept
{
    for (auto &subList : _subLists)
    {
        subList->NotifyEvent(eventIdentifier, argument);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タスクへのイベント通知
 *  \param[in]  identifier          タスクID
 *  \param[in]  eventIdentifier     イベントID
 *  \param[in]  argument            引数
 */
/* ------------------------------------------------------------------------- */
void NodeList::NotifyEvent(Identifier identifier, EventIdentifier eventIdentifier, void *argument) noexcept
{
    if (auto *subList = GetSubList(identifier); subList != nullptr)
    {
        subList->NotifyEvent(eventIdentifier, argument);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      サブリストを取得
 *  \param[in]  identifier  タスクID
 *  \return     IDに対応したサブリスト．見つからない場合はnullptr
 */
/* ------------------------------------------------------------------------- */
NodeSubList *NodeList::GetSubList(Identifier identifier) const noexcept
{
    for (const auto &subList : _subLists)
    {
        if (subList->GetIdentifier() == identifier)
        {
            return subList.get();
        }
        else if (subList->GetIdentifier() >= identifier)
        {
            break;
        }
    }

    return nullptr;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      サブリストを生成
 *  \param[in]  identifier  タスクID
 *  \return     生成したサブリストのアドレス
 */
/* ------------------------------------------------------------------------- */
NodeSubList *NodeList::CreateSubList(Identifier identifier) noexcept
{
    // サブリストを走査して適切な挿入位置に挿入
    for (auto it = _subLists.begin(); it != _subLists.end(); ++it)
    {
        // 既に存在する場合は作る必要がない
        if ((*it)->GetIdentifier() == identifier)
        {
            return it->get();
        }
        // 挿入するIDより大きいIDのサブリストが見つかったら，その前の位置に挿入
        else if ((*it)->GetIdentifier() > identifier)
        {
            return _subLists.insert(it, STL::make_unique<NodeSubList>(identifier))->get();
        }
    }

    // 上記のループが実行されなかったら末尾に追加
    _subLists.push_back(STL::make_unique<NodeSubList>(identifier));
    return _subLists.rbegin()->get();
}
}    // namespace MGL::Task

// vim: et ts=4 sw=4 sts=4
