// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_window_module_win32.cc
 *  \brief      MGL Win32用ウィンドウモジュール
 *  \date       Since: March 29, 2021. 16:56:35 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/system/window/mgl_window_module_win32.h>
#if defined(MGL_TARGET_WIN32)

#include <atlstr.h>

#if defined(MGL_RENDERER_ENABLE_D3D11)
#include <mgl/render/d3d11/mgl_d3d11_device.h>
#endif

#include <mgl/platform/win32/mgl_win32_window.h>

namespace MGL::System
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      サイズの変更
 *  \param[in]  width   幅
 *  \param[in]  height  高さ
 */
/* ------------------------------------------------------------------------- */
void Win32WindowModule::Resize(uint32_t width, uint32_t height) noexcept
{
    auto &window = Win32::Window::GetInstance();

    window.Resize(static_cast<int>(width), static_cast<int>(height));
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フルスクリーンの設定
 *  \param[in]  isEnabled   有効フラグ
 */
/* ------------------------------------------------------------------------- */
void Win32WindowModule::SetFullscreen(bool isEnabled) noexcept
{
    if (isEnabled != IsFullscreen())
    {
        ToggleFullscreen();
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フルスクリーンの有効・無効の切り替え
 */
/* ------------------------------------------------------------------------- */
void Win32WindowModule::ToggleFullscreen() noexcept
{
#if defined(MGL_RENDERER_ENABLE_D3D11)
    auto &device = Render::D3D11::Device::GetInstance();
    device.SetFullscreen(!device.IsFullscreen());
#endif
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フルスクリーン状態の取得
 *  \retval     true    フクスクリーン
 *  \retval     false   ウィンドウ
 */
/* ------------------------------------------------------------------------- */
bool Win32WindowModule::IsFullscreen() const noexcept
{
#if defined(MGL_RENDERER_ENABLE_D3D11)
    auto &device = Render::D3D11::Device::GetInstance();
    return device.IsFullscreen();
#else
    return false;
#endif
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ウィンドウにフォーカスを設定
 */
/* ------------------------------------------------------------------------- */
void Win32WindowModule::Focus() noexcept
{
    auto &window = Win32::Window::GetInstance();

    SetActiveWindow(window.GetWindowHandler());
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フォーカス状態の取得
 *  \retval     true    フォーカス
 *  \retval     false   フォーカスでない
 */
/* ------------------------------------------------------------------------- */
bool Win32WindowModule::IsFocused() const noexcept
{
    auto &window = Win32::Window::GetInstance();

    return (GetActiveWindow() == window.GetWindowHandler());
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アラートの表示
 *  \param[in]  message     表示メッセージ
 */
/* ------------------------------------------------------------------------- */
void Win32WindowModule::Alert(const char *message) const noexcept
{
    auto &window = Win32::Window::GetInstance();

    constexpr size_t kBufferSize = 4096;
    wchar_t wStr[kBufferSize];

    MultiByteToWideChar(CP_UTF8, 0, message, -1, wStr, kBufferSize);

    MessageBox(window.GetWindowHandler(), wStr, L"Alert", MB_OK | MB_ICONEXCLAMATION);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ウィンドウタイトルの設定
 *  \param[in]  title   設定するウィンドウタイトル
 */
/* ------------------------------------------------------------------------- */
void Win32WindowModule::SetTitle(const char *title) noexcept
{
    auto &window = Win32::Window::GetInstance();

    window.SetWindowTitle(title);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ウィンドウの識別子を取得
 *  \return     ウィンドウの識別子
 */
/* ------------------------------------------------------------------------- */
int32_t Win32WindowModule::GetIdentifier() const noexcept
{
    return 0;    // 未対応
}


/* ------------------------------------------------------------------------- */
/*!
    *  \brief      ウィンドウが存在している画面のサイズ情報を取得
    *  \return     画面サイズ
    */
/* ------------------------------------------------------------------------- */
ScreenSize Win32WindowModule::GetScreenSize() const noexcept
{
    ScreenSize screenSize;

    auto &window = MGL::Win32::Window::GetInstance();

    auto *monitor = MonitorFromWindow(window.GetWindowHandler(), MONITOR_DEFAULTTONEAREST);

    MONITORINFOEX monitorInfo = {};
    monitorInfo.cbSize = sizeof(MONITORINFOEX);
    GetMonitorInfo(monitor, &monitorInfo);

    screenSize.physicalSize.x = static_cast<float>(monitorInfo.rcMonitor.right - monitorInfo.rcMonitor.left);
    screenSize.physicalSize.y = static_cast<float>(monitorInfo.rcMonitor.bottom - monitorInfo.rcMonitor.top);
    screenSize.backingSize = screenSize.physicalSize;

    screenSize.logicalSize = screenSize.physicalSize / window.GetLogicalScaleFactor();

    return screenSize;
}

}    // namespace MGL::System

#endif    // MGL_TARGET_WIN32
          // vim: et ts=4 sw=4 sts=4
