// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_window_module_macos.mm
 *  \brief      MGL macOS用ウィンドウ制御
 *  \date       Since: February 19, 2021. 15:16:58 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/system/window/mgl_window_module_macos.h>
#if defined(MGL_TARGET_MACOS)

#include <AppKit/AppKit.h>

#include <mgl/event/mgl_event.h>
#include <mgl/input/mouse/mgl_mouse.h>
#include <mgl/platform/apple/macos/mgl_macos_view_controller.h>

namespace MGL::System
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
MacOSWindowModule::MacOSWindowModule() noexcept
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      サイズの変更
 *  \param[in]  width   幅
 *  \param[in]  height  高さ
 */
/* ------------------------------------------------------------------------- */
void MacOSWindowModule::Resize(uint32_t width, uint32_t height) noexcept
{
    if (!NSThread.isMainThread)
    {
        dispatch_sync(dispatch_get_main_queue(), ^() { Resize(width, height); });
        return;
    }

    auto *mainWindow = MGLmacOSViewController.mainWindow;
    if (mainWindow == nullptr)
    {
        return;
    }

    auto frame = [mainWindow convertRectFromBacking:CGRectMake(0, 0, width, height)];
    frame = [mainWindow frameRectForContentRect:frame];
    frame.origin = mainWindow.frame.origin;
    [mainWindow setFrame:frame
                 display:YES
                 animate:YES];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フルスクリーンの設定
 *  \param[in]  isEnabled   有効フラグ
 */
/* ------------------------------------------------------------------------- */
void MacOSWindowModule::SetFullscreen(bool isEnabled) noexcept
{
    if (isEnabled != IsFullscreen())
    {
        ToggleFullscreen();
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フルスクリーンの有効・無効の切り替え
 */
/* ------------------------------------------------------------------------- */
void MacOSWindowModule::ToggleFullscreen() noexcept
{
    if (!NSThread.isMainThread)
    {
        dispatch_sync(dispatch_get_main_queue(), ^() { ToggleFullscreen(); });
        return;
    }

    auto *mainWindow = MGLmacOSViewController.mainWindow;
    if (mainWindow != nullptr)
    {
        [mainWindow toggleFullScreen:nullptr];
        MGL::Event::Notify(MGL::Event::NotifyType::ShouldClearInput);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フルスクリーン状態の取得
 *  \retval     true    フクスクリーン
 *  \retval     false   ウィンドウ
 */
/* ------------------------------------------------------------------------- */
bool MacOSWindowModule::IsFullscreen() const noexcept
{
    if (!NSThread.isMainThread)
    {
        bool isFullscreen = false;
        auto *p = &isFullscreen;
        dispatch_sync(dispatch_get_main_queue(), ^() { *p = IsFullscreen(); });
        return isFullscreen;
    }

    auto *mainWindow = MGLmacOSViewController.mainWindow;
    if (mainWindow == nullptr)
    {
        return false;
    }

    if ((mainWindow.styleMask & NSWindowStyleMaskFullScreen) != 0)
    {
        return true;
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ウィンドウにフォーカスを設定
 */
/* ------------------------------------------------------------------------- */
void MacOSWindowModule::Focus() noexcept
{
    if (!NSThread.isMainThread)
    {
        dispatch_sync(dispatch_get_main_queue(), ^() { Focus(); });
        return;
    }

    auto *mainWindow = MGLmacOSViewController.mainWindow;
    if (mainWindow == nullptr)
    {
        return;
    }

    [NSApplication.sharedApplication activateIgnoringOtherApps:true];
    [mainWindow makeKeyWindow];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フォーカス状態の取得
 *  \retval     true    フォーカス
 *  \retval     false   フォーカスでない
 */
/* ------------------------------------------------------------------------- */
bool MacOSWindowModule::IsFocused() const noexcept
{
    if (!NSThread.isMainThread)
    {
        bool isFocusd = false;
        auto *p = &isFocusd;
        dispatch_sync(dispatch_get_main_queue(), ^() { *p = IsFocused(); });
        return isFocusd;
    }

    auto *mainWindow = MGLmacOSViewController.mainWindow;
    if (mainWindow == nullptr)
    {
        return false;
    }

    return mainWindow.isKeyWindow;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アラートの表示
 *  \param[in]  message     表示メッセージ
 */
/* ------------------------------------------------------------------------- */
void MacOSWindowModule::Alert(const char *message) const noexcept
{
    if (!NSThread.isMainThread)
    {
        dispatch_sync(dispatch_get_main_queue(), ^() { Alert(message); });
        return;
    }

    NSAlert *alert = [NSAlert new];
    alert.messageText = [NSString stringWithUTF8String:message];
    [alert runModal];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ウィンドウタイトルの設定
 *  \param[in]  title   設定するウィンドウタイトル
 */
/* ------------------------------------------------------------------------- */
void MacOSWindowModule::SetTitle(const char *title) noexcept
{
    if (!NSThread.isMainThread)
    {
        dispatch_sync(dispatch_get_main_queue(), ^() { SetTitle(title); });
        return;
    }

    auto *mainWindow = MGLmacOSViewController.mainWindow;
    if (mainWindow != nullptr)
    {
        [mainWindow setTitle:[NSString stringWithUTF8String:title]];
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ウィンドウの識別子を取得
 *  \return     ウィンドウの識別子
 */
/* ------------------------------------------------------------------------- */
int32_t MacOSWindowModule::GetIdentifier() const noexcept
{
    if (!NSThread.isMainThread)
    {
        int32_t identifier = 0;
        auto *p = &identifier;
        dispatch_sync(dispatch_get_main_queue(), ^() { *p = GetIdentifier(); });
        return identifier;
    }

    return static_cast<int32_t>(MGLmacOSViewController.mainWindow.windowNumber);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ウィンドウが存在している画面のサイズ情報を取得
 *  \return     画面サイズ
 */
/* ------------------------------------------------------------------------- */
ScreenSize MacOSWindowModule::GetScreenSize() const noexcept
{
    if (!NSThread.isMainThread)
    {
        ScreenSize info;
        auto *p = &info;
        dispatch_sync(dispatch_get_main_queue(), ^() { *p = GetScreenSize(); });
        return info;
    }

    ScreenSize info;

    // ウィンドウを取得
    auto *mainWindow = MGLmacOSViewController.mainWindow;
    if (mainWindow == nullptr)
    {
        return info;
    }

    // ウィンドウからディスプレイIDを取得
    auto displayId = [mainWindow.screen.deviceDescription[@"NSScreenNumber"] unsignedIntValue];

    // ディスプレイの物理解像度
    auto displayModes = CGDisplayCopyAllDisplayModes(displayId, nullptr);
    auto size = CFArrayGetCount(displayModes);
    for (long i = 0; i < size; i++)
    {
        CGDisplayModeRef mode = (CGDisplayModeRef)CFArrayGetValueAtIndex(displayModes, i);

        if (CGDisplayModeGetIOFlags(mode) & kDisplayModeNativeFlag)
        {
            info.physicalSize.x = static_cast<float>(CGDisplayModeGetPixelWidth(mode));
            info.physicalSize.y = static_cast<float>(CGDisplayModeGetPixelHeight(mode));
            break;
        }
    }
    CFRelease(displayModes);

    // 論理解像度を取得
    auto logicalFrame = mainWindow.screen.frame;
    info.logicalSize.x = static_cast<float>(logicalFrame.size.width);
    info.logicalSize.y = static_cast<float>(logicalFrame.size.height);

    // 内部解像度を取得
    auto backingFrame = [mainWindow.screen convertRectToBacking:logicalFrame];
    info.backingSize.x = static_cast<float>(backingFrame.size.width);
    info.backingSize.y = static_cast<float>(backingFrame.size.height);

    info.isAvailable = true;

    return info;
}
}    // namespace MGL::System

#endif    // MGL_TARGET_MACOS \
          // vim: et ts=4 sw=4 sts=4
