// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_locale_module_win32.cc
 *  \brief      MGL Win32用ロケール情報モジュール
 *  \date       Since: March 29, 2021. 17:44:46 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/system/locale/mgl_locale_module_win32.h>
#if defined(MGL_TARGET_WIN32)

#include <Windows.h>

#include <mgl/text/mgl_text_converter.h>

namespace MGL::System
{
namespace
{
// 言語テーブル
struct LanguageTable
{
    Language language;        // 言語
    const wchar_t *prefix;    // 判定用プリフィックス（言語-書体）
    const wchar_t *suffix;    // 判定用サフィックス（国名）
};
constexpr LanguageTable kLanguageTable[] =
    {
        {Language::BrazilPortuguese, L"pt-BR", L"-BR"},         // ブラジルポルトガル語
        {Language::LatinAmericaSpanish, L"es-419", L"-419"},    // ラテンアメリカのスペイン語
        {Language::LatinAmericaSpanish, L"es-", L"-CR"},        // ラテンアメリカのスペイン語（コスタリカ）
        {Language::LatinAmericaSpanish, L"es-", L"-MX"},        // ラテンアメリカのスペイン語（メキシコ）
        {Language::LatinAmericaSpanish, L"es-", L"-GT"},        // ラテンアメリカのスペイン語（グアテマラ）
        {Language::LatinAmericaSpanish, L"es-", L"-HN"},        // ラテンアメリカのスペイン語（ホンジュラス）
        {Language::LatinAmericaSpanish, L"es-", L"-SV"},        // ラテンアメリカのスペイン語（エルサルバドル）
        {Language::LatinAmericaSpanish, L"es-", L"-NI"},        // ラテンアメリカのスペイン語（ニカラグア）
        {Language::LatinAmericaSpanish, L"es-", L"-PA"},        // ラテンアメリカのスペイン語（パナマ）
        {Language::LatinAmericaSpanish, L"es-", L"-CU"},        // ラテンアメリカのスペイン語（キューバ）
        {Language::LatinAmericaSpanish, L"es-", L"-DO"},        // ラテンアメリカのスペイン語（ドミニカ共和国）
        {Language::LatinAmericaSpanish, L"es-", L"-VE"},        // ラテンアメリカのスペイン語（ベネズエラ）
        {Language::LatinAmericaSpanish, L"es-", L"-CO"},        // ラテンアメリカのスペイン語（コロンビア）
        {Language::LatinAmericaSpanish, L"es-", L"-EC"},        // ラテンアメリカのスペイン語（エクアドル）
        {Language::LatinAmericaSpanish, L"es-", L"-PE"},        // ラテンアメリカのスペイン語（ペルー）
        {Language::LatinAmericaSpanish, L"es-", L"-BO"},        // ラテンアメリカのスペイン語（ボリビア）
        {Language::LatinAmericaSpanish, L"es-", L"-PY"},        // ラテンアメリカのスペイン語（パラグアイ）
        {Language::LatinAmericaSpanish, L"es-", L"-CL"},        // ラテンアメリカのスペイン語（チリ）
        {Language::LatinAmericaSpanish, L"es-", L"-UY"},        // ラテンアメリカのスペイン語（ウルグアイ）
        {Language::LatinAmericaSpanish, L"es-", L"-AR"},        // ラテンアメリカのスペイン語（アルゼンチン）
        {Language::LatinAmericaSpanish, L"es-", L"-PR"},        // ラテンアメリカのスペイン語（プエルトリコ）
        {Language::LatinAmericaSpanish, L"es-", L"-BM"},        // ラテンアメリカのスペイン語（バミューダ）

        {Language::SimplifiedChinese, L"zh-Hans", nullptr},     // 簡体中文
        {Language::TraditionalChinese, L"zh-Hant", nullptr},    // 繁体中文

        {Language::Arabic, L"ar-", nullptr},        // アラビア語
        {Language::Bulgarian, L"bg-", nullptr},     // ブルガリア語
        {Language::Czech, L"cs-", nullptr},         // チェコ語
        {Language::Dansk, L"da-", nullptr},         // デンマーク語
        {Language::Deutsch, L"de-", nullptr},       // ドイツ語
        {Language::English, L"en-", nullptr},       // 英語
        {Language::French, L"fr-", nullptr},        // フランス語
        {Language::Greek, L"el-", nullptr},         // ギリシャ語
        {Language::Italiano, L"it-", nullptr},      // イタリア語
        {Language::Japanese, L"ja-", nullptr},      // 日本語
        {Language::Korean, L"ko-", nullptr},        // 韓国語
        {Language::Magyar, L"hu-", nullptr},        // ハンガリー語
        {Language::Nederlands, L"nl-", nullptr},    // オランダ語
        {Language::Norsk, L"no-", nullptr},         // ノルウェー語
        {Language::Norsk, L"nn-", nullptr},         // ノルウェー語（ニーノシュク）
        {Language::Norsk, L"nb-", nullptr},         // ノルウェー語（ブークモール）
        {Language::Polski, L"pl-", nullptr},        // ポーランド語
        {Language::Portuguese, L"pt-", nullptr},    // ポルトガル語
        {Language::Romanian, L"ro-", nullptr},      // ルーマニア語
        {Language::Russian, L"ru-", nullptr},       // ロシア語
        {Language::Spanish, L"es-", nullptr},       // スペイン語
        {Language::Suomi, L"fi-", nullptr},         // フィンランド語
        {Language::Svenska, L"sv-", nullptr},       // スウェーデン語
        {Language::Thai, L"th-", nullptr},          // タイ語
        {Language::Turkish, L"tr-", nullptr},       // トルコ語
        {Language::Ukrainian, L"uk-", nullptr},     // ウクライナ語
        {Language::Vietnamese, L"vi-", nullptr},    // ベトナム語
};


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テキストの先頭が一致しているかを取得
 *  \param[in]  text    検索対象のテキスト
 *  \param[in]  prefix  検索する先頭文字列
 *  \retval     true    先頭が一致している
 *  \retval     false   先頭が一致していない
 *  \note
 *      C++20以降であればwstring::starts_with()が使用できる．
 */
/* ------------------------------------------------------------------------- */
bool StartsWith(const wchar_t *text, const wchar_t *prefix)
{
    size_t index = 0;

    while ((text[index] != '\0') && (prefix[index] != '\0'))
    {
        if (text[index] != prefix[index])
        {
            return false;
        }

        index++;
    }

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テキストの末尾が一致しているかを取得
 *  \param[in]  text    検索対象のテキスト
 *  \param[in]  suffix  検索する先頭文字列
 *  \retval     true    末尾が一致している
 *  \retval     false   末尾が一致していない
 *  \note
 *      C++20以降であればwstring::ends_with()が使用できる．
 */
/* ------------------------------------------------------------------------- */
bool EndsWith(const wchar_t *text, const wchar_t *suffix)
{
    size_t textIndex = lstrlenW(text);
    size_t suffixIndex = lstrlenW(suffix);

    if ((textIndex == 0) || (suffixIndex == 0))
    {
        return false;
    }

    textIndex--;
    suffixIndex--;

    while (true)
    {
        if (text[textIndex] != suffix[suffixIndex])
        {
            return false;
        }

        if ((textIndex == 0) || (suffixIndex == 0))
        {
            break;
        }

        textIndex--;
        suffixIndex--;
    }

    return true;
}
}    // namespace


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
Win32LocaleModule::Win32LocaleModule() noexcept
    : _currentInfo()
    , _defaultInfo()
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化
 */
/* ------------------------------------------------------------------------- */
void Win32LocaleModule::Initialize() noexcept
{
    // 言語を取得
    wchar_t localeName[LOCALE_NAME_MAX_LENGTH];
    if (GetUserDefaultLocaleName(localeName, LOCALE_NAME_MAX_LENGTH) > 0)
    {
        for (const auto &tableElem : kLanguageTable)
        {
            // プリフィックスが一致しているかをチェック
            if (StartsWith(localeName, tableElem.prefix))
            {
                // サフィックスが指定されている場合は末尾もチェック
                if (tableElem.suffix != nullptr)
                {
                    if (EndsWith(localeName, tableElem.suffix))
                    {
                        _defaultInfo.language = tableElem.language;
                    }
                }
                // 指定されていなければ確定
                else
                {
                    _defaultInfo.language = tableElem.language;
                }
            }
        }
    }

    constexpr size_t kBufferSize = 256;
    wchar_t localeInfo[kBufferSize];

    // 小数点数の区切り文字を取得
    if (GetLocaleInfoEx(localeName, LOCALE_SDECIMAL, localeInfo, kBufferSize) > 0)
    {
        MGL::Text::ToUTF8(_currentInfo.decimalSeparator, localeInfo, kBufferSize * sizeof(wchar_t), MGL::Text::Encoding::UTF16LE);
    }

    // 整数の区切り文字を取得
    if (GetLocaleInfoEx(localeName, LOCALE_STHOUSAND, localeInfo, kBufferSize) > 0)
    {
        MGL::Text::ToUTF8(_currentInfo.groupingSeparator, localeInfo, kBufferSize * sizeof(wchar_t), MGL::Text::Encoding::UTF16LE);
    }

    // 時差を取得
    TIME_ZONE_INFORMATION timeZoneInfo;
    if (GetTimeZoneInformation(&timeZoneInfo) != TIME_ZONE_ID_INVALID)
    {
        _currentInfo.timeDifference = timeZoneInfo.Bias * -60;
    }

    _currentInfo = _defaultInfo;
}

}    // namespace MGL::System

#endif    // MGL_TARGET_WIN32

// vim: et ts=4 sw=4 sts=4
