// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_locale_module_apple.mm
 *  \brief      MGL Apple（macOS/iOS/tvOS）ロケール情報モジュール
 *  \date       Since: February 23, 2021. 11:28:42 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/system/locale/mgl_locale_module_apple.h>
#if defined(MGL_TARGET_MACOS) || defined(MGL_TARGET_IOS) || defined(MGL_TARGET_TVOS)

#include <Foundation/Foundation.h>

namespace MGL::System
{

namespace
{
// 言語テーブル
struct LanguageTable
{
    Language language;      // 言語
    const char *prefix;     // 判定用プリフィックス
};
constexpr LanguageTable kLanguageTable[] =
{
    {Language::BrazilPortuguese,    "pt-BR"},   // ブラジルポルトガル語
    {Language::LatinAmericaSpanish, "es-419"},  // ラテンアメリカのスペイン語
    {Language::LatinAmericaSpanish, "es-CR"},   // ラテンアメリカのスペイン語（コスタリカ）
    {Language::LatinAmericaSpanish, "es-MX"},   // ラテンアメリカのスペイン語（メキシコ）
    {Language::LatinAmericaSpanish, "es-GT"},   // ラテンアメリカのスペイン語（グアテマラ）
    {Language::LatinAmericaSpanish, "es-HN"},   // ラテンアメリカのスペイン語（ホンジュラス）
    {Language::LatinAmericaSpanish, "es-SV"},   // ラテンアメリカのスペイン語（エルサルバドル）
    {Language::LatinAmericaSpanish, "es-NI"},   // ラテンアメリカのスペイン語（ニカラグア）
    {Language::LatinAmericaSpanish, "es-PA"},   // ラテンアメリカのスペイン語（パナマ）
    {Language::LatinAmericaSpanish, "es-CU"},   // ラテンアメリカのスペイン語（キューバ）
    {Language::LatinAmericaSpanish, "es-DO"},   // ラテンアメリカのスペイン語（ドミニカ共和国）
    {Language::LatinAmericaSpanish, "es-VE"},   // ラテンアメリカのスペイン語（ベネズエラ）
    {Language::LatinAmericaSpanish, "es-CO"},   // ラテンアメリカのスペイン語（コロンビア）
    {Language::LatinAmericaSpanish, "es-EC"},   // ラテンアメリカのスペイン語（エクアドル）
    {Language::LatinAmericaSpanish, "es-PE"},   // ラテンアメリカのスペイン語（ペルー）
    {Language::LatinAmericaSpanish, "es-BO"},   // ラテンアメリカのスペイン語（ボリビア）
    {Language::LatinAmericaSpanish, "es-PY"},   // ラテンアメリカのスペイン語（パラグアイ）
    {Language::LatinAmericaSpanish, "es-CL"},   // ラテンアメリカのスペイン語（チリ）
    {Language::LatinAmericaSpanish, "es-UY"},   // ラテンアメリカのスペイン語（ウルグアイ）
    {Language::LatinAmericaSpanish, "es-AR"},   // ラテンアメリカのスペイン語（アルゼンチン）
    {Language::LatinAmericaSpanish, "es-PR"},   // ラテンアメリカのスペイン語（プエルトリコ）
    {Language::LatinAmericaSpanish, "es-BM"},   // ラテンアメリカのスペイン語（バミューダ）

    {Language::SimplifiedChinese,   "zh-Hans"}, // 簡体中文
    {Language::TraditionalChinese,  "zh-Hant"}, // 繁体中文

    {Language::Arabic,              "ar-"},     // アラビア語
    {Language::Bulgarian,           "bg-"},     // ブルガリア語
    {Language::Czech,               "cs-"},     // チェコ語
    {Language::Dansk,               "da-"},     // デンマーク語
    {Language::Deutsch,             "de-"},     // ドイツ語
    {Language::English,             "en-"},     // 英語
    {Language::French,              "fr-"},     // フランス語
    {Language::Greek,               "el-"},     // ギリシャ語
    {Language::Italiano,            "it-"},     // イタリア語
    {Language::Japanese,            "ja-"},     // 日本語
    {Language::Korean,              "ko-"},     // 韓国語
    {Language::Magyar,              "hu-"},     // ハンガリー語
    {Language::Nederlands,          "nl-"},     // オランダ語
    {Language::Norsk,               "no-"},     // ノルウェー語
    {Language::Norsk,               "nn-"},     // ノルウェー語（ニーノシュク）
    {Language::Norsk,               "nb-"},     // ノルウェー語（ブークモール）
    {Language::Polski,              "pl-"},     // ポーランド語
    {Language::Portuguese,          "pt-"},     // ポルトガル語
    {Language::Romanian,            "ro-"},     // ルーマニア語
    {Language::Russian,             "ru-"},     // ロシア語
    {Language::Spanish,             "es-"},     // スペイン語
    {Language::Suomi,               "fi-"},     // フィンランド語
    {Language::Svenska,             "sv-"},     // スウェーデン語
    {Language::Thai,                "th-"},     // タイ語
    {Language::Turkish,             "tr-"},     // トルコ語
    {Language::Ukrainian,           "uk-"},     // ウクライナ語
    {Language::Vietnamese,          "vi-"},     // ベトナム語
};
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
AppleLocaleModule::AppleLocaleModule() noexcept
    : _currentInfo()
    , _defaultInfo()
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化
 */
/* ------------------------------------------------------------------------- */
void AppleLocaleModule::Initialize() noexcept
{
    // 言語の取得
    for (const NSString *language in NSLocale.preferredLanguages)
    {
        for (const auto tableElem : kLanguageTable)
        {
            if ([language hasPrefix:@(tableElem.prefix)])
            {
                _defaultInfo.language = tableElem.language;
                break;
            }
        }

        if (_defaultInfo.language != Language::Unknown)
        {
            break;
        }
    }

    // 数値の区切り文字を取得
    auto *locale = NSLocale.autoupdatingCurrentLocale;
    _defaultInfo.decimalSeparator = locale.decimalSeparator.UTF8String;
    _defaultInfo.groupingSeparator = locale.groupingSeparator.UTF8String;


    // 時差を取得
    _defaultInfo.timeDifference = static_cast<int32_t>(NSTimeZone.localTimeZone.secondsFromGMT);

    _currentInfo = _defaultInfo;
}
}    // namespace MGL::System

#endif    // defined(MGL_TARGET_MACOS) || defined(MGL_TARGET_IOS) || defined(MGL_TARGET_TVOS)

// vim: et ts=4 sw=4 sts=4
