// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_debug_module_win32.cc
 *  \brief      MGL Win32用モジュール
 *  \date       Since: March 29, 2021. 16:08:01 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/system/debug/mgl_debug_module_win32.h>
#if defined(MGL_TARGET_WIN32)

#include <cstdarg>
#include <cstdio>

#include <Windows.h>

namespace MGL::System
{
namespace
{
constexpr int kTraceBufferSize = 1024;
}
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  logLevel    ログレベル
 */
/* ------------------------------------------------------------------------- */
Win32DebugModule::Win32DebugModule(LogLevel logLevel) noexcept
    : _logLevel(logLevel)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ログ出力
 *  \param[in]  logLevel    ログレベル
 *  \param[in]  format      フォーマット文字列
 */
/* ------------------------------------------------------------------------- */
void Win32DebugModule::Log(LogLevel logLevel, const char *format, ...) const noexcept
{
    if (logLevel > _logLevel)
    {
        return;
    }

    va_list arg = nullptr;
    char text[kTraceBufferSize];
    va_start(arg, format);
    if (vsnprintf(text, kTraceBufferSize, format, arg) >= kTraceBufferSize)
    {
        text[kTraceBufferSize - 1] = '\0';
    }
    va_end(arg);

    OutputDebugStringA(text);
    OutputDebugStringA("\n");
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      プログラムの中断（再開可能）
 */
/* ------------------------------------------------------------------------- */
void Win32DebugModule::Trap() const noexcept
{
    DebugBreak();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      プログラムの中断（再開不可）
 */
/* ------------------------------------------------------------------------- */
void Win32DebugModule::Abort() const noexcept
{
    abort();
}
}    // namespace MGL::System

#endif    // MGL_TARGET_WIN32

// vim: et ts=4 sw=4 sts=4
