// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_debug_module_apple.mm
 *  \brief      MGL Apple用デバッグモジュール
 *  \date       Since: February 25, 2021. 11:30:16 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/system/debug/mgl_debug_module_apple.h>
#if defined(MGL_TARGET_MACOS) || defined(MGL_TARGET_IOS) || defined(MGL_TARGET_TVOS)

#include <cstdarg>
#include <cstdio>

#include <Foundation/Foundation.h>
#include <TargetConditionals.h>

namespace MGL::System
{
namespace
{
constexpr int kTraceBufferSize = 1024;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  logLevel    ログレベル
 */
/* ------------------------------------------------------------------------- */
AppleDebugModule::AppleDebugModule(LogLevel logLevel) noexcept
    : _logLevel(logLevel)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ログ出力
 *  \param[in]  logLevel    ログレベル
 *  \param[in]  format      フォーマット文字列
 */
/* ------------------------------------------------------------------------- */
void AppleDebugModule::Log(LogLevel logLevel, const char *format, ...) const noexcept
{
    if (logLevel > _logLevel)
    {
        return;
    }

    va_list arg;
    char text[kTraceBufferSize];
    va_start(arg, format);
    if (vsnprintf(text, kTraceBufferSize, format, arg) >= kTraceBufferSize)
    {
        text[kTraceBufferSize - 1] = '\0';
    }
    va_end(arg);

    fprintf((logLevel <= LogLevel::Warning) ? stderr : stdout, "%s\n", text);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      プログラムの中断（再開可能）
 */
/* ------------------------------------------------------------------------- */
void AppleDebugModule::Trap() const noexcept
{
    raise(SIGTRAP);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      プログラムの中断（再開不可）
 */
/* ------------------------------------------------------------------------- */
void AppleDebugModule::Abort() const noexcept
{
    abort();
}
}    // namespace MGL::System
#endif    // defined(MGL_TARGET_MACOS) || defined(MGL_TARGET_IOS) || defined(MGL_TARGET_TVOS)

// vim: et ts=4 sw=4 sts=4
