// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_chrono_module_win32.cc
 *  \brief      MGL Win32用の日付と時刻のモジュール
 *  \date       Since: March 29, 2021. 18:18:48 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/system/chrono/mgl_chrono_module_win32.h>
#if defined(MGL_TARGET_WIN32)

#include <ctime>

#include <Windows.h>

#include <mgl/system/mgl_system_locale.h>

namespace MGL::System
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      現在時刻をエポック時間で取得
 *  \return     現在時刻
 */
/* ------------------------------------------------------------------------- */
EpochTime Win32ChronoModule::GetCurrentEpochTime() const noexcept
{
    return time(nullptr);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      Tickタイマーに対応しているかを取得
 *  \retval     true    対応している
 *  \retval     false   対応していない
 */
/* ------------------------------------------------------------------------- */
bool Win32ChronoModule::IsAvailableTickTimer() const noexcept
{
    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      現在のTickTimeを取得
 *  \return     現在のTickTime
 */
/* ------------------------------------------------------------------------- */
TickTime Win32ChronoModule::GetTickTime() const noexcept
{
    LARGE_INTEGER currentTick;
    QueryPerformanceCounter(&currentTick);

    return static_cast<TickTime>(currentTick.QuadPart);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      TickTimeをナノ秒に変換
 *  \return     変換後の時間
 */
/* ------------------------------------------------------------------------- */
float Win32ChronoModule::TicksToNanoseconds(TickTime tickTime) const noexcept
{
    LARGE_INTEGER frequency;
    QueryPerformanceFrequency(&frequency);

    auto time = static_cast<double>(tickTime) / static_cast<double>(frequency.QuadPart);
    time *= 1000'000'000.0;

    return static_cast<float>(time);
}


}    // namespace MGL::System

#endif    // MGL_TARGET_WIN32

// vim: et ts=4 sw=4 sts=4
