// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_chrono_elapsed_timer.cc
 *  \brief      MGL 経過タイマー
 *  \date       Since: February 24, 2021. 6:27:38 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/system/chrono/mgl_chrono_elapsed_timer.h>

namespace MGL::System
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
ElapsedTimer::ElapsedTimer() noexcept
    : _tick(Chrono().GetTickTime())
    , _lappedTimeNanosecond(0.0f)
{
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  tickTime    TickTimeの初期値
 *  \note
 *      MGL初期化前にコンストラクタを呼びたい場合はこちらを使用．
 *      引数に0を指定すると，初回のLap()呼び出し時に値が初期化され，
 *      2回目移行の呼び出しで結果を更新する．
 */
/* ------------------------------------------------------------------------- */
ElapsedTimer::ElapsedTimer(TickTime tickTime) noexcept
    : _tick(tickTime)
    , _lappedTimeNanosecond(0.0f)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      測定時間を更新
 */
/* ------------------------------------------------------------------------- */
void ElapsedTimer::Lap() noexcept
{
    const Chrono chrono;

    if (_tick == 0)
    {
        _tick = chrono.GetTickTime();
    }
    else
    {
        auto prevTick = _tick;
        _tick = chrono.GetTickTime();

        _lappedTimeNanosecond = chrono.TicksToNanoseconds(chrono.GetTickPassTime(prevTick, _tick));
    }
}

}    // namespace MGL::System

// vim: et ts=4 sw=4 sts=4
