// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_application_module_win32.cc
 *  \brief      MGL Win32用アプリケーションモジュール
 *  \date       Since: March 29, 2021. 16:31:06 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/system/application/mgl_application_module_win32.h>
#if defined(MGL_TARGET_WIN32)

#include <Windows.h>
#if defined(min)
#undef min
#endif

#include <mgl/platform/win32/mgl_win32_window.h>

namespace MGL::System
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
Win32ApplicationModule::Win32ApplicationModule() noexcept
    : _isCaffeinated(false)
{
    // コマンドライン引数を展開
    int argc = 0;
    auto *argv = CommandLineToArgvW(L"", &argc);
    for (int i = 0; i < argc; ++i)
    {
        constexpr size_t kBufferSize = 4096;
        char arg[kBufferSize];
        size_t length = 0;
        if (auto error = wcstombs_s(&length, arg, kBufferSize, argv[i], kBufferSize); error == 0)
        {
            _argumentArray.emplace_back(arg);
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      スリープの抑制
 *  \param[in]  isEnabled   有効フラグ
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool Win32ApplicationModule::Caffeinate(bool isEnabled) noexcept
{
    // 設定
    if (isEnabled)
    {
        SetThreadExecutionState(ES_CONTINUOUS | ES_SYSTEM_REQUIRED | ES_DISPLAY_REQUIRED);
    }
    // 解除
    else
    {
        SetThreadExecutionState(ES_CONTINUOUS);
    }

    _isCaffeinated = isEnabled;

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      スリープの抑制状態を取得
 *  \retval     true    有効
 *  \retval     false   無効
 */
/* ------------------------------------------------------------------------- */
bool Win32ApplicationModule::IsCaffeinated() const noexcept
{
    return _isCaffeinated;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      実行引数を取得
 *  \return     実行引数の配列
 */
/* ------------------------------------------------------------------------- */
const ArgumentArray &Win32ApplicationModule::GetArguments() const noexcept
{
    return _argumentArray;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アプリケーションの終了
 */
/* ------------------------------------------------------------------------- */
void Win32ApplicationModule::Quit() noexcept
{
    PostQuitMessage(0);
}


/* ------------------------------------------------------------------------- */
/*!
    *  \brief      クリップボードにテキストをコピー
    *  \param[in]  text    コピーするテキスト
    *  \retval     true    成功
    *  \retval     false   失敗
    */
/* ------------------------------------------------------------------------- */
bool Win32ApplicationModule::SetClipboardText(const char *text) noexcept
{
    auto size = strlen(text);
    if (size == 0)
    {
        return false;
    }
    size++;

    if (!OpenClipboard(MGL::Win32::Window::GetInstance().GetWindowHandler()))
    {
        return false;
    }

    bool isSucceeded = false;

    if (auto *global = GlobalAlloc(GHND | GMEM_SHARE, size); global != nullptr)
    {
        if (auto *p = GlobalLock(global); p != nullptr)
        {
            memcpy(p, text, size);
            GlobalUnlock(global);

            EmptyClipboard();
            SetClipboardData(CF_TEXT, global);

            isSucceeded = true;
        }
    }

    CloseClipboard();

    return isSucceeded;
}


/* ------------------------------------------------------------------------- */
/*!
    *  \brief      クリップボードからテキストを取得
    *  \param[out] text    テキストのコピー先
    *  \retval     true    成功
    *  \retval     false   失敗
    */
/* ------------------------------------------------------------------------- */
bool Win32ApplicationModule::GetClipboardText(STL::string &text) noexcept
{
    if (!OpenClipboard(MGL::Win32::Window::GetInstance().GetWindowHandler()))
    {
        return false;
    }

    bool isSucceeded = false;

    if (auto *global = GetClipboardData(CF_TEXT); global != nullptr)
    {
        constexpr size_t kBufferSize = 256;    // NOTE: セキュリティリスクに繋がるのでこれ以上は読み取らない
        char temporary[kBufferSize];

        if (auto *p = GlobalLock(global); p != nullptr)
        {
            auto readSize = std::min(static_cast<size_t>(GlobalSize(global)), kBufferSize - 1);
            memcpy(temporary, p, readSize);
            temporary[readSize + 1] = '\0';

            text = temporary;
            isSucceeded = true;

            GlobalUnlock(global);
        }

        CloseClipboard();
    }

    return isSucceeded;
}

}    // namespace MGL::System

#endif    // MGL_TARGET_WIN32

// vim: et ts=4 sw=4 sts=4
