// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_application_module_macos.mm
 *  \brief      MGL macOS用アプリケーションモジュール
 *  \date       Since: February 22, 2021. 10:47:54 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/system/application/mgl_application_module_macos.h>
#if defined(MGL_TARGET_MACOS)

#include <AppKit/AppKit.h>
#include <Foundation/Foundation.h>

namespace MGL::System
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
MacOSApplicationModule::MacOSApplicationModule() noexcept
    : _argumentArray()
    , _caffeinateAssertionID(kIOPMNullAssertionID)
{
    // 実行引数を取得
    for (const NSString *touch in NSProcessInfo.processInfo.arguments)
    {
        _argumentArray.push_back(touch.UTF8String);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      スリープの抑制
 *  \param[in]  isEnabled   有効フラグ
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool MacOSApplicationModule::Caffeinate(bool isEnabled) noexcept
{
    // 抑制
    if (isEnabled)
    {
        if (_caffeinateAssertionID == kIOPMNullAssertionID)
        {
            IOReturn success = IOPMAssertionCreateWithName(kIOPMAssertionTypeNoDisplaySleep, kIOPMAssertionLevelOn, CFSTR("Caffeinate"), &_caffeinateAssertionID);
            if (success != kIOReturnSuccess)
            {
                _caffeinateAssertionID = kIOPMNullAssertionID;
                return false;
            }
        }
    }
    // 抑制解除
    else
    {
        if (_caffeinateAssertionID != kIOPMNullAssertionID)
        {
            IOReturn success = IOPMAssertionRelease(_caffeinateAssertionID);
            if (success != kIOReturnSuccess)
            {
                return false;
            }
            _caffeinateAssertionID = kIOPMNullAssertionID;
        }
    }

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      スリープの抑制状態を取得
 *  \retval     true    有効
 *  \retval     false   無効
 */
/* ------------------------------------------------------------------------- */
bool MacOSApplicationModule::IsCaffeinated() const noexcept
{
    return _caffeinateAssertionID != kIOPMNullAssertionID;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      実行引数を取得
 *  \return     実行引数の配列
 */
/* ------------------------------------------------------------------------- */
const ArgumentArray &MacOSApplicationModule::GetArguments() const noexcept
{
    return _argumentArray;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アプリケーションの終了
 */
/* ------------------------------------------------------------------------- */
void MacOSApplicationModule::Quit() noexcept
{
    [NSApplication.sharedApplication terminate:nil];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      クリップボードにテキストをコピー
 *  \param[in]  text    コピーするテキスト
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool MacOSApplicationModule::SetClipboardText(const char *text) noexcept
{
    NSPasteboard *pasteboard = [NSPasteboard generalPasteboard];
    [pasteboard clearContents];
    bool succeed = [pasteboard setString:[NSString stringWithUTF8String:text] forType:NSPasteboardTypeString];

    return succeed;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      クリップボードからテキストを取得
 *  \param[out] text    テキストのコピー先
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool MacOSApplicationModule::GetClipboardText(STL::string &text) noexcept
{
    NSPasteboard *pasteboard = [NSPasteboard generalPasteboard];
    NSString *basteboardText = [pasteboard stringForType:NSPasteboardTypeString];

    text = STL::string([basteboardText UTF8String]);

    return true;
}
}    // namespace MGL::System

#endif    // MGL_TARGET_MACOS

// vim: et ts=4 sw=4 sts=4
