// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_application_module_common.cc
 *  \brief      MGL 共通アプリケーションモジュール
 *  \date       Since: February 22, 2021. 11:54:10 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/system/application/mgl_application_module_common.h>

#include <thread>

namespace MGL::System
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
CommonApplicationModule::CommonApplicationModule() noexcept
    : _eventPreFrameUpdate(Event::NotifyType::PreFrameUpdate, OnEventPreFrameUpdate, this)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フレーム更新前のイベント処理
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       未使用
 */
/* ------------------------------------------------------------------------- */
void CommonApplicationModule::OnEventPreFrameUpdate(void *callbackArg, [[maybe_unused]] void *notifyArg) noexcept
{
    auto *thisPtr = static_cast<CommonApplicationModule *>(callbackArg);

    thisPtr->UpdateFPS();

    thisPtr->_elapsedTimer.Lap();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      前回のフレームからの経過時間を取得
 *  \return     経過時間（ミリ秒）
 */
/* ------------------------------------------------------------------------- */
float CommonApplicationModule::GetFrameElapsedTime() const noexcept
{
    return _elapsedTimer.GetLappedTimeMillisecond();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      FPSを取得
 *  \return     FPS（過去1秒間に実際に更新された回数）
 */
/* ------------------------------------------------------------------------- */
uint32_t CommonApplicationModule::GetFramesPerSecond() const noexcept
{
    return _fps;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      FPSの更新処理
 */
/* ------------------------------------------------------------------------- */
void CommonApplicationModule::UpdateFPS() noexcept
{
    _frameUpdateCount++;

    // 秒単位の時間が変化していたらFPSを更新
    auto currentTime = Chrono().GetCurrentEpochTime();
    if (currentTime != _prevUpdateTime)
    {
        _fps = _frameUpdateCount;
        _frameUpdateCount = 0;
        _prevUpdateTime = currentTime;
    }
}

}    // namespace MGL::System

// vim: et ts=4 sw=4 sts=4
