// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_savedata_server.cc
 *  \brief      MGL セーブデータサーバ
 *  \date       Since: August 5, 2021. 9:18:03 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/savedata/mgl_savedata_server.h>

#include <mgl/file/mgl_file.h>
#include <mgl/hash/mgl_hash_fnv1a.h>
#include <mgl/system/mgl_system_debug_macro.h>

namespace MGL::Savedata
{
namespace
{
constexpr uint32_t kChunkEndMarker = Hash::FNV1a("EndOfChunk");
}
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      インスタンスの取得
 *  \return     インスタンスの参照
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<Server> &Server::GetInstanceRef() noexcept
{
    static STL::unique_ptr<Server> sInstance = nullptr;
    return sInstance;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
Server::Server() noexcept
    : _state(State::PreInitialize)
    , _existError(false)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      セーブデータサーバの初期化
 *  \param[in]  delegate   使用するデリゲート
 *  \retval     0          成功
 *  \retval     0以外の正数  デリゲートの初期化エラー（意味はデリゲート側で定義）
 *  \retval     負数        サーバ側の要因によるエラー（現状は-1のみ，他は予約）
 */
/* ------------------------------------------------------------------------- */
int32_t Server::Initialize(STL::unique_ptr<ServerDelegate> &delegate) noexcept
{
    // 多重初期化防止
    if (_state != State::PreInitialize)
    {
        return -1;
    }

    // デリゲートを設定
    if (delegate == nullptr)
    {
        return -1;
    }
    _delegate = std::move(delegate);

    // デリゲートを初期化
    auto result = _delegate->OnInitialize();
    if (result != 0)
    {
        return result;
    }

    _state = State::Ready;

    return 0;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      チャンクの追加
 *  \param[in]  identifier      追加先のセーブデータ識別子
 *  \param[in]  chunk           追加するチャンク
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool Server::AddChunk(DataIdentifier identifier, Chunk *chunk) noexcept
{
    // 処理中は追加できない
    if (!IsReady())
    {
        return false;
    }

    auto &chunkList = _chunkListMap[identifier];

    chunkList.push_back(chunk);

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      チャンクの削除
 *  \param[in]  identifier      削除するセーブデータ識別子
 *  \param[in]  chunk           削除するチャンク
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool Server::RemoveChunk(DataIdentifier identifier, Chunk *chunk) noexcept
{
    if (_chunkListMap.find(identifier) == _chunkListMap.end())
    {
        return false;
    }

    auto &chunkList = _chunkListMap[identifier];
    for (auto it = chunkList.begin(); it != chunkList.end(); ++it)
    {
        if (*it == chunk)
        {
            chunkList.erase(it);
            return true;
        }
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      セーブ要求
 *  \param[in]  async           非同期フラグ
 *  \param[in]  requests        要求するデータの配列
 *  \param[in]  haltOnError     trueを指定した場合，エラー発生時点で処理を中断する
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool Server::SaveRequest(const STL::vector<RequestInfo> &requests, bool async, bool haltOnError) noexcept
{
    // 処理中は実行できない
    if (!IsReady())
    {
        return false;
    }

    // 各種パラメータを初期化
    _state = State::Saving;
    _existError = false;
    _lastResults.clear();
    _requests = requests;

    // 保存処理
    auto saveProcess = [this, haltOnError]()
    {
        for (const auto &argument : _requests)
        {
            auto result = Save(argument.identifier, argument.index);
            _lastResults.push_back(result);
            if (result)
            {
                _existError = true;
                if (haltOnError)
                {
                    break;
                }
            }
        }

        _state = State::Ready;

        return !_existError;
    };

    // 引数に応じて非同期 or 同期で実行
    if (async)
    {
        _future = std::async(std::launch::async, saveProcess);
    }
    else
    {
        saveProcess();
    }

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ロード要求
 *  \param[in]  requests                要求するデータの配列
 *  \param[in]  async                   非同期フラグ
 *  \param[in]  haltOnError             trueを指定した場合，エラー発生時点で処理を中断する
 *  \param[in]  errorOnFileNotExist     trueを指定した場合，ファイルが存在しない場合をエラーとして扱う
 *  \retval     true                    成功
 *  \retval     false                   失敗
 */
/* ------------------------------------------------------------------------- */
bool Server::LoadRequest(const STL::vector<RequestInfo> &requests, bool async, bool haltOnError, bool errorOnFileNotExist) noexcept
{
    // 処理中は実行できない
    if (!IsReady())
    {
        return false;
    }

    // 各種パラメータを初期化
    _state = State::Loading;
    _existError = false;
    _lastResults.clear();
    _requests = requests;

    // 読み込み処理
    auto loadProcess = [this, haltOnError, errorOnFileNotExist]()
    {
        for (const auto &argument : _requests)
        {
            auto result = Load(argument.identifier, argument.index);
            _lastResults.push_back(result);
            if (result.HasError(errorOnFileNotExist))
            {
                _existError = true;
                if (haltOnError)
                {
                    break;
                }
            }
        }

        _state = State::Ready;

        return !_existError;
    };

    // 引数に応じて非同期 or 同期で実行
    if (async)
    {
        _future = std::async(std::launch::async, loadProcess);
    }
    else
    {
        loadProcess();
    }

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      セーブ
 *  \param[in]  identifier  保存するデータの識別子
 *  \param[in]  index       インデックス（用途はデリゲート側で定義）
 *  \return     処理結果
 */
/* ------------------------------------------------------------------------- */
Result Server::Save(DataIdentifier identifier, uint32_t index) noexcept
{
    // デリゲートがnullptrなら失敗
    if (_delegate == nullptr)
    {
        return {identifier, Error::DelegateNotReady};
    }

    // セーブデータの識別子が存在していなければ失敗
    if (_chunkListMap.find(identifier) == _chunkListMap.end())
    {
        return {identifier, Error::IdentifierNotFound};
    }

    // デリゲートのアクセス前処理を実行
    if (!_delegate->OnPreAccess(identifier, index, AccessType::Save))
    {
        return {identifier, Error::DelegateHasError};
    }

    // ファイル情報を取得してバイナリストリームを準備
    const auto *fileInfo = _delegate->GetFileInfo(identifier, index);
    if (fileInfo == nullptr)
    {
        return {identifier, Error::DelegateHasError};
    }
    auto buffer = STL::make_unique<std::byte[]>(fileInfo->size);
    ByteStream stream(buffer.get(), fileInfo->size);
    stream.Clear(ByteStream::ClearType::Zero);

    // 各チャンクを走査してストリームに格納
    auto &chunkList = _chunkListMap[identifier];
    for (auto &chunk : chunkList)
    {
        // ヘッダ部分
        stream.Write(chunk->GetSavedataChunkIdentifier(identifier));
        stream.Write(chunk->GetSavedataChunkVersion(identifier));
        auto *sizeWritePoint = static_cast<uint32_t *>(stream.GetCurrentPoint());
        stream.Skip(sizeof(uint32_t));

        // チャンククラスのデータ収集関数を呼び出す
        auto chunkStartPoint = stream.GetOffset();
        chunk->OnSaving(stream, identifier);
        *sizeWritePoint = static_cast<uint32_t>(stream.GetOffset() - chunkStartPoint);

        // チャンク終了マーカーの書き込み
        stream.Write(kChunkEndMarker);

        // オーバーフローしていたら失敗
        if (stream.IsOverflowed())
        {
            return {identifier, Error::BufferNotEnough};
        }
    }

    // ファイルに書き込み
    try
    {
        File::ThrowingHandle file(fileInfo->path, File::OpenMode::Write);
        file.Write(buffer.get(), stream.GetOffset());
    }
    catch (const File::Exception &fileException)
    {
        MGL_ERROR(fileException.what());
        return {identifier, Error::FailedToFileAccess, fileException.GetError()};
    }

    // デリゲートのアクセス後処理を実行
    if (!_delegate->OnPostAccess(identifier, index, AccessType::Save))
    {
        return {identifier, Error::DelegateHasError};
    }

    return {identifier, Error::None};
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ロード
 *  \param[in]  identifier  読み込むデータの識別子
 *  \param[in]  index       インデックス（用途はデリゲート側で定義）
 *  \return     処理結果
 */
/* ------------------------------------------------------------------------- */
Result Server::Load(DataIdentifier identifier, uint32_t index) noexcept
{
    // デリゲートがnullptrなら失敗
    if (_delegate == nullptr)
    {
        return {identifier, Error::DelegateNotReady};
    }

    // セーブデータの識別子が存在していなければ失敗
    if (_chunkListMap.find(identifier) == _chunkListMap.end())
    {
        return {identifier, Error::IdentifierNotFound};
    }

    // デリゲートのアクセス前処理を実行
    if (!_delegate->OnPreAccess(identifier, index, AccessType::Load))
    {
        return {identifier, Error::DelegateHasError};
    }

    // ファイル情報を取得して読み込み
    const auto *fileInfo = _delegate->GetFileInfo(identifier, index);
    if (fileInfo == nullptr)
    {
        return {identifier, Error::DelegateHasError};
    }

    STL::unique_ptr<std::byte[]> buffer = nullptr;
    size_t fileSize = 0;
    try
    {
        File::ThrowingHandle file(fileInfo->path);
        fileSize = file.GetSize();
        buffer = STL::make_unique<std::byte[]>(fileSize);
        file.Read(buffer.get(), fileSize);
    }
    catch (const File::Exception &fileException)
    {
        MGL_ERROR(fileException.what());
        return {identifier, Error::FailedToFileAccess, fileException.GetError()};
    }

    // バイナリストリームを準備
    ByteStream stream(static_cast<const void *>(buffer.get()), fileSize);

    // 登録されたチャンクを走査して一致するものがあれば読み込み
    auto chunkList = _chunkListMap[identifier];
    while (stream.GetOffset() < stream.GetSize())
    {
        auto chunkId = stream.Read<ChunkIdentifier>();
        auto version = stream.Read<uint32_t>();
        auto chunkSize = static_cast<size_t>(stream.Read<uint32_t>());
        bool found = false;

        for (auto it = chunkList.begin(); it != chunkList.end(); ++it)
        {
            auto *chunk = *it;

            if (chunk->GetSavedataChunkIdentifier(identifier) == chunkId)
            {
                if (!chunk->OnLoading(stream, identifier, version, chunkSize))
                {
                    return {identifier, Error::FailedToChunkParse};
                }
                chunkList.erase(it);
                found = true;
                break;
            }
        }

        // 不明なチャンクはスキップ
        if (!found)
        {
            stream.Skip(chunkSize);
        }

        // チャンク処理後に終了マーカーが無ければパースに失敗している
        if (auto marker = stream.Read<uint32_t>(); marker != kChunkEndMarker)
        {
            return {identifier, Error::MarkerNotMatching};
        }
    }

    // デリゲートのアクセス後処理を実行
    if (!_delegate->OnPostAccess(identifier, index, AccessType::Load))
    {
        return {identifier, Error::DelegateHasError};
    }

    return {identifier, Error::None};
}
}    // namespace MGL::Savedata
// vim: et ts=4 sw=4 sts=4
