// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_savedata_chunk_register.cc
 *  \brief      MGL セーブデータチャンク登録クラス
 *  \date       Since: August 11, 2021. 7:20:22 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/savedata/mgl_savedata_chunk_register.h>

#include <mgl/savedata/mgl_savedata_server.h>

namespace MGL::Savedata
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  chunk   登録するチャンク
 */
/* ------------------------------------------------------------------------- */
ChunkRegister::ChunkRegister(Chunk *chunk) noexcept
    : _chunk(chunk)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デストラクタ
 */
/* ------------------------------------------------------------------------- */
ChunkRegister::~ChunkRegister() noexcept
{
    if (Server::HasInstance())
    {
        auto &server = Server::GetInstance();

        for (auto identifier : _identifiers)
        {
            server.RemoveChunk(identifier, _chunk);
        }

        _identifiers.clear();
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      登録
 *  \param[in]  identifier  登録するセーブデータの識別子
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool ChunkRegister::Register(DataIdentifier identifier) noexcept
{
    if (Server::HasInstance())
    {
        auto &server = Server::GetInstance();

        if (server.AddChunk(identifier, _chunk))
        {
            _identifiers.push_back(identifier);
            return true;
        }
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      登録解除
 *  \param[in]  identifier  登録解除するセーブデータの識別子
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool ChunkRegister::Unregister(DataIdentifier identifier) noexcept
{
    if (Server::HasInstance())
    {
        auto &server = Server::GetInstance();

        if (server.AddChunk(identifier, _chunk))
        {
            _identifiers.remove(identifier);
            return true;
        }
    }

    return false;
}
}    // namespace MGL::Savedata

// vim: et ts=4 sw=4 sts=4
