// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_savedata.cc
 *  \brief      MGL セーブデータ
 *  \date       Since: August 11, 2021. 10:54:33 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/savedata/mgl_savedata.h>

namespace MGL::Savedata
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      セーブ要求
 *  \param[in]  requests        要求するデータの配列
 *  \param[in]  haltOnError     trueを指定した場合，エラー発生時点で処理を中断する
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool SaveRequests(const STL::vector<RequestInfo> &requests, bool haltOnError) noexcept
{
    if (auto *server = GetServer(); server != nullptr)
    {
        return server->SaveRequest(requests, true, haltOnError);
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      セーブ要求
 *  \param[in]  identifier      要求するデータの識別子
 *  \param[in]  index           要求するデータのインデックス
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool SaveRequests(DataIdentifier identifier, uint32_t index) noexcept
{
    if (auto *server = GetServer(); server != nullptr)
    {
        const RequestInfo argument(identifier, index);
        return server->SaveRequest({argument}, true, true);
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ロード要求
 *  \param[in]  requests                要求するデータの配列
 *  \param[in]  haltOnError             trueを指定した場合，エラー発生時点で処理を中断する
 *  \param[in]  errorOnFileNotExist     trueを指定した場合，ファイルが存在しない場合をエラーとして扱う
 *  \retval     true                    成功
 *  \retval     false                   失敗
 */
/* ------------------------------------------------------------------------- */
bool LoadRequests(const STL::vector<RequestInfo> &requests, bool haltOnError, bool errorOnFileNotExist) noexcept
{
    if (auto *server = GetServer(); server != nullptr)
    {
        return server->LoadRequest(requests, true, haltOnError, errorOnFileNotExist);
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ロード要求
 *  \param[in]  identifier              要求するデータの識別子
 *  \param[in]  index                   要求するデータのインデックス
 *  \param[in]  errorOnFileNotExist     trueを指定した場合，ファイルが存在しない場合をエラーとして扱う
 *  \retval     true                    成功
 *  \retval     false                   失敗
 */
/* ------------------------------------------------------------------------- */
bool LoadRequests(DataIdentifier identifier, uint32_t index, bool errorOnFileNotExist) noexcept
{
    if (auto *server = GetServer(); server != nullptr)
    {
        const RequestInfo argument(identifier, index);
        return server->LoadRequest({argument}, true, true, errorOnFileNotExist);
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      同期セーブ
 *  \param[in]  requests        要求するデータの配列
 *  \param[in]  haltOnError     trueを指定した場合，エラー発生時点で処理を中断する
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool SaveSynchronously(const STL::vector<RequestInfo> &requests, bool haltOnError) noexcept
{
    if (auto *server = GetServer(); server != nullptr)
    {
        return server->SaveRequest(requests, false, haltOnError);
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      同期セーブ
 *  \param[in]  identifier      要求するデータの識別子
 *  \param[in]  index           要求するデータのインデックス
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool SaveSynchronously(DataIdentifier identifier, uint32_t index) noexcept
{
    if (auto *server = GetServer(); server != nullptr)
    {
        const RequestInfo argument(identifier, index);
        return server->SaveRequest({argument}, false, true);
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      同期ロード
 *  \param[in]  requests                要求するデータの配列
 *  \param[in]  haltOnError             trueを指定した場合，エラー発生時点で処理を中断する
 *  \param[in]  errorOnFileNotExist     trueを指定した場合，ファイルが存在しない場合をエラーとして扱う
 *  \retval     true                    成功
 *  \retval     false                   失敗
 */
/* ------------------------------------------------------------------------- */
bool LoadSynchronously(const STL::vector<RequestInfo> &requests, bool haltOnError, bool errorOnFileNotExist) noexcept
{
    if (auto *server = GetServer(); server != nullptr)
    {
        return server->LoadRequest(requests, false, haltOnError, errorOnFileNotExist);
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      同期ロード
 *  \param[in]  identifier              要求するデータの識別子
 *  \param[in]  index                   要求するデータのインデックス
 *  \param[in]  errorOnFileNotExist     trueを指定した場合，ファイルが存在しない場合をエラーとして扱う
 *  \retval     true                    成功
 *  \retval     false                   失敗
 */
/* ------------------------------------------------------------------------- */
bool LoadSynchronously(DataIdentifier identifier, uint32_t index, bool errorOnFileNotExist) noexcept
{
    if (auto *server = GetServer(); server != nullptr)
    {
        const RequestInfo argument(identifier, index);
        return server->LoadRequest({argument}, false, true, errorOnFileNotExist);
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ロード中かを取得
 *  \retval     true    ロード中
 *  \retval     false   ロード中でない
 */
/* ------------------------------------------------------------------------- */
bool IsLoading() noexcept
{
    if (auto *server = GetServer(); server != nullptr)
    {
        return server->IsLoading();
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      セーブ中かを取得
 *  \retval     true    セーブ中
 *  \retval     false   セーブ中でない
 */
/* ------------------------------------------------------------------------- */
bool IsSaving() noexcept
{
    if (auto *server = GetServer(); server != nullptr)
    {
        return server->IsSaving();
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      セーブ・ロードが可能な状態かを取得
 *  \retval     true    可能
 *  \retval     false   不可能（処理中 or 未初期化）
 */
/* ------------------------------------------------------------------------- */
bool IsReady() noexcept
{
    if (auto *server = GetServer(); server != nullptr)
    {
        return server->IsReady();
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      処理中かを取得
 *  \retval     true    処理中
 *  \retval     false   処理中でない
 */
/* ------------------------------------------------------------------------- */
bool IsProcessing() noexcept
{
    if (auto *server = GetServer(); server != nullptr)
    {
        return server->IsProcessing();
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      前回の処理に成功したかを取得
 *  \retval     true    成功
 *  \retval     false   失敗 or 処理中
 */
/* ------------------------------------------------------------------------- */
bool IsSucceeded() noexcept
{
    if (auto *server = GetServer(); server != nullptr)
    {
        return server->IsSucceeded();
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      最後に発生した処理結果を取得
 *  \return     処理結果
 */
/* ------------------------------------------------------------------------- */
STL::vector<Result> GetLastResults() noexcept
{
    if (auto *server = GetServer(); server != nullptr)
    {
        return server->GetLastResults();
    }

    return {};
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      実行中の処理が完了するまで待機
 *  \retval     true    処理に成功
 *  \retval     false   処理に失敗
 */
/* ------------------------------------------------------------------------- */
bool Wait() noexcept
{
    if (auto *server = GetServer(); server != nullptr)
    {
        return server->Wait();
    }

    return false;
}
}    // namespace MGL::Savedata

// vim: et ts=4 sw=4 sts=4
