// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_texture_storage.cc
 *  \brief      MGL テクスチャストレージ
 *  \date       Since: December 10, 2020. 16:03:23 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

/*
 *  メモ:
 *      非同期読み込みの処理はそのうちキューイングする方式に変更する予定．
 *      今のままだと同時に大量に実行した際にスレッドリソースが枯渇して失敗すると思われる．
 *      （スレッドリソースが枯渇するほど大量のテクスチャを読み込むのかは別として）
 *
 *  2024-06-03追記:
 *      そもそもstd::threadじゃなくてstd::asyncにした方が良い。
 */

#include <mgl/render/mgl_texture_storage.h>

#include <mgl/mgl_environment.h>
#include <mgl/system/mgl_system_debug_macro.h>

#include <thread>

namespace MGL::Render
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  generator       テクスチャジェネレータ
 */
/* ------------------------------------------------------------------------- */
TextureStorage::TextureStorage(STL::unique_ptr<TextureGenerator> generator) noexcept
    : _textureGenerator(std::move(generator))
    , _loadingCount(0)
{
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デストラクタ
 */
/* ------------------------------------------------------------------------- */
TextureStorage::~TextureStorage() noexcept
{
    WaitLoading();

    _resources.clear();
    _loaders.clear();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャをメモリ上のイメージデータから読み込み
 *  \param[in]  imageData   メモリ上のイメージデータ
 *  \param[in]  dataSize    データサイズ
 *  \param[in]  loaderKey   テクスチャローダのキー
 *  \return     読み込んだテクスチャリソース．失敗時はnullptr
 */
/* ------------------------------------------------------------------------- */
SharedTextureResource TextureStorage::Load(const void *imageData, size_t dataSize, TextureLoaderKey loaderKey) noexcept
{
    // テクスチャローダーを取得
    auto *loader = GetTextureLoader(loaderKey);
    if (loader == nullptr)
    {
        return nullptr;
    }

    // 派生テクスチャリソースを生成
    auto resource = CreateDerivedResource();
    if (resource == nullptr)
    {
        return nullptr;
    }

    // ローダーに読み込ませる
    if (!loader->Load(*resource, imageData, dataSize))
    {
        return nullptr;
    }

    return resource;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャをメモリ上のイメージデータから非同期で読み込み
 *  \param[in]  imageData   メモリ上のイメージデータ
 *  \param[in]  dataSize    データサイズ
 *  \param[in]  loaderKey   テクスチャローダのキー
 *  \return     新たに生成されたテクスチャリソース
 *  \note
 *      成否はIsLoading()がfalseを返すようになった後，IsValid()の結果でチェックする
 */
/* ------------------------------------------------------------------------- */
SharedTextureResource TextureStorage::LoadAsync(const void *imageData, size_t dataSize, TextureLoaderKey loaderKey) noexcept
{
    // 派生テクスチャリソースを生成
    auto resource = CreateDerivedResource();
    if (resource == nullptr)
    {
        return nullptr;
    }

    // テクスチャローダーを取得
    auto *loader = GetTextureLoader(loaderKey);
    if (loader == nullptr)
    {
        return nullptr;
    }

    // 読み込みスレッドを実行
    SetLoading(resource.get(), true);
    std::thread thread([this, loader, resource, imageData, dataSize]
    {
        loader->Load(*resource, imageData, dataSize);
        SetLoading(resource.get(), false);
    });
    if (thread.get_id() == std::thread::id())
    {
        MGL_WARNING("[TextureStorage] Failed to create a thread.");
        SetLoading(resource.get(), false);
    }
    else
    {
        thread.detach();
    }

    return resource;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャをファイルから読み込み
 *  \param[in]  imagePath   画像ファイルのパス
 *  \param[in]  loaderKey   テクスチャローダのキー
 *  \return     読み込んだテクスチャリソース．失敗時はnullptr
 */
/* ------------------------------------------------------------------------- */
SharedTextureResource TextureStorage::Load(const File::PathView &imagePath, TextureLoaderKey loaderKey) noexcept
{
    // テクスチャローダーを取得
    auto *loader = GetTextureLoader(loaderKey);
    if (loader == nullptr)
    {
        return nullptr;
    }

    // 派生テクスチャリソースを生成
    auto resource = CreateDerivedResource();
    if (resource == nullptr)
    {
        return nullptr;
    }

    // ローダーに読み込ませる
    if (!loader->Load(*resource, imagePath))
    {
        return nullptr;
    }

    return resource;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャを画像ファイルから非同期で読み込み
 *  \param[in]  imagePath   画像ファイルのパス
 *  \param[in]  loaderKey   テクスチャローダのキー
 *  \return     新たに生成されたテクスチャリソース
 *  \note
 *      成否はIsLoading()がfalseを返すようになった後，IsValid()の結果でチェックする
 */
/* ------------------------------------------------------------------------- */
SharedTextureResource TextureStorage::LoadAsync(const File::PathView &imagePath, TextureLoaderKey loaderKey) noexcept
{
    // 派生テクスチャリソースを生成
    auto resource = CreateDerivedResource();
    if (resource == nullptr)
    {
        return nullptr;
    }

    // テクスチャローダーを取得
    auto *loader = GetTextureLoader(loaderKey);
    if (loader == nullptr)
    {
        return nullptr;
    }

    // 読み込みスレッドを実行
    SetLoading(resource.get(), true);
    std::thread thread([this, loader, resource, imagePath]
    {
        loader->Load(*resource, imagePath);
        SetLoading(resource.get(), false);
    });
    if (thread.get_id() == std::thread::id())
    {
        MGL_WARNING("[TextureStorage] Failed to create a thread.");
        SetLoading(resource.get(), false);
    }
    else
    {
        thread.detach();
    }

    return resource;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャをメモリ上のイメージデータから読み込んでテーブルに登録
 *  \param[in]  key         テクスチャリソースのキー
 *  \param[in]  imageData   メモリ上のイメージデータ
 *  \param[in]  dataSize    データサイズ
 *  \param[in]  loaderKey   テクスチャローダのキー
 *  \return     読み込んだテクスチャリソース．失敗時はnullptr
 */
/* ------------------------------------------------------------------------- */
SharedTextureResource TextureStorage::Load(TextureKey key, const void *imageData, size_t dataSize, TextureLoaderKey loaderKey) noexcept
{
    // 既に生成されている場合はそれを返す
    auto resource = Get(key);
    if (resource != nullptr)
    {
        return resource;
    }

    // 読み込み
    resource = Load(imageData, dataSize, loaderKey);
    if (resource == nullptr)
    {
        return nullptr;
    }

    // リソースをテーブルに追加
    const std::scoped_lock lock(_tableMutex);
    _resources.emplace(key, resource);

    return resource;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャをメモリ上のイメージデータから非同期で読み込んでテーブルに登録
 *  \param[in]  key         テクスチャリソースのキー
 *  \param[in]  imageData   メモリ上のイメージデータ
 *  \param[in]  dataSize    データサイズ
 *  \param[in]  loaderKey   テクスチャローダのキー
 *  \return     新たに生成されたテクスチャリソース
 *  \note
 *      成否はIsLoading()がfalseを返すようになった後，IsValid()の結果でチェックする
 */
/* ------------------------------------------------------------------------- */
SharedTextureResource TextureStorage::LoadAsync(TextureKey key, const void *imageData, size_t dataSize, TextureLoaderKey loaderKey) noexcept
{
    // 既に生成されている場合はそれを返す
    auto resource = Get(key);
    if (resource != nullptr)
    {
        return resource;
    }

    // 派生テクスチャリソースを生成
    resource = CreateDerivedResource();
    if (resource == nullptr)
    {
        return nullptr;
    }

    // テクスチャローダーを取得
    auto *loader = GetTextureLoader(loaderKey);
    if (loader == nullptr)
    {
        return nullptr;
    }

    // 先にテーブルに追加
    _tableMutex.lock();
    _resources.emplace(key, resource);
    _tableMutex.unlock();

    // 読み込みスレッドを実行
    SetLoading(resource.get(), true);
    std::thread thread([this, loader, resource, imageData, dataSize, key]
    {
        if (!loader->Load(*resource, imageData, dataSize))
        {
            Destroy(key);
        }
        SetLoading(resource.get(), false);
    });
    if (thread.get_id() == std::thread::id())
    {
        MGL_WARNING("[TextureStorage] Failed to create a thread.");
        Destroy(key);
        SetLoading(resource.get(), false);
    }
    else
    {
        thread.detach();
    }

    return resource;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャをファイルから読み込んでテーブルに登録
 *  \param[in]  key         テクスチャリソースのキー
 *  \param[in]  imagePath   画像ファイルのパス
 *  \param[in]  loaderKey   テクスチャローダのキー
 *  \return     読み込んだテクスチャリソース．失敗時はnullptr
 */
/* ------------------------------------------------------------------------- */
SharedTextureResource TextureStorage::Load(TextureKey key, const File::PathView &imagePath, TextureLoaderKey loaderKey) noexcept
{
    // 既に生成されている場合はそれを返す
    auto resource = Get(key);
    if (resource != nullptr)
    {
        return resource;
    }

    // 読み込み
    resource = Load(imagePath, loaderKey);
    if (resource == nullptr)
    {
        return nullptr;
    }

    // リソースをテーブルに追加
    const std::scoped_lock lock(_tableMutex);
    _resources.emplace(key, resource);

    return resource;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャを画像ファイルから非同期で読み込んでテーブルに登録
 *  \param[in]  key         テクスチャリソースのキー
 *  \param[in]  imagePath   画像ファイルのパス
 *  \param[in]  loaderKey   テクスチャローダのキー
 *  \return     新たに生成されたテクスチャリソース
 *  \note
 *      成否はIsLoading()がfalseを返すようになった後，IsValid()の結果でチェックする
 */
/* ------------------------------------------------------------------------- */
SharedTextureResource TextureStorage::LoadAsync(TextureKey key, const File::PathView &imagePath, TextureLoaderKey loaderKey) noexcept
{
    // 既に生成されている場合はそれを返す
    auto resource = Get(key);
    if (resource != nullptr)
    {
        return resource;
    }

    // 派生テクスチャリソースを生成
    resource = CreateDerivedResource();
    if (resource == nullptr)
    {
        return nullptr;
    }

    // テクスチャローダーを取得
    auto *loader = GetTextureLoader(loaderKey);
    if (loader == nullptr)
    {
        return nullptr;
    }

    // 先にテーブルに追加
    _tableMutex.lock();
    _resources.emplace(key, resource);
    _tableMutex.unlock();

    // 読み込みスレッドを実行
    SetLoading(resource.get(), true);
    std::thread thread([this, loader, resource, imagePath, key]
    {
        if (!loader->Load(*resource, imagePath))
        {
            Destroy(key);
        }
        SetLoading(resource.get(), false);
    });
    if (thread.get_id() == std::thread::id())
    {
        MGL_WARNING("[TextureStorage] Failed to create a thread.");
        Destroy(key);
        SetLoading(resource.get(), false);
    }
    else
    {
        thread.detach();
    }

    return resource;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャリソースの生成
 *  \param[in]  key             生成するテクスチャリソースのキー
 *  \param[in]  pixelData       ピクセルデータ
 *  \param[in]  pixelFormat     ピクセルフォーマット
 *  \param[in]  width           幅
 *  \param[in]  height          高さ
 *  \return     生成されたテクスチャリソース．失敗時にはnullptr．
 */
/* ------------------------------------------------------------------------- */
SharedTextureResource TextureStorage::Create(TextureKey key, const void *pixelData, PixelFormat pixelFormat, uint32_t width, uint32_t height) noexcept
{
    // 既に生成されている場合はそれを返す
    auto resource = Get(key);
    if (resource != nullptr)
    {
        return resource;
    }

    // 派生テクスチャリソースを生成
    resource = CreateDerivedResource();
    if (resource == nullptr)
    {
        return nullptr;
    }

    // 通常のテクスチャを生成
    if (!resource->Create(pixelData, pixelFormat, width, height))
    {
        return nullptr;
    }

    // リソースをテーブルに追加
    const std::scoped_lock lock(_tableMutex);
    _resources.emplace(key, resource);

    return resource;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャリソースの生成
 *  \param[in]  pixelData       ピクセルデータ
 *  \param[in]  pixelFormat     ピクセルフォーマット
 *  \param[in]  width           幅
 *  \param[in]  height          高さ
 *  \return     生成されたテクスチャリソース．失敗時にはnullptr．
 */
/* ------------------------------------------------------------------------- */
SharedTextureResource TextureStorage::Create(const void *pixelData, PixelFormat pixelFormat, uint32_t width, uint32_t height) noexcept
{
    // 派生テクスチャリソースを生成
    auto resource = CreateDerivedResource();
    if (resource == nullptr)
    {
        return nullptr;
    }

    // 通常のテクスチャを生成
    if (!resource->Create(pixelData, pixelFormat, width, height))
    {
        return nullptr;
    }

    return resource;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      レンダーターゲットの生成
 *  \param[in]  key             生成するテクスチャリソースのキー
 *  \param[in]  width           幅
 *  \param[in]  height          高さ
 *  \return     生成されたテクスチャリソース．失敗時にはnullptr．
 */
/* ------------------------------------------------------------------------- */
SharedTextureResource TextureStorage::CreateRenderTarget(TextureKey key, uint32_t width, uint32_t height) noexcept
{
    // 既に生成されている場合はそれを返す
    auto resource = Get(key);
    if (resource != nullptr)
    {
        // 生成済みのテクスチャがレンダーターゲットでなければ失敗
        if (!resource->IsRenderTarget())
        {
            return nullptr;
        }

        return resource;
    }

    // 派生テクスチャリソースを生成
    resource = CreateDerivedResource();
    if (resource == nullptr)
    {
        return nullptr;
    }

    // レンダーターゲットを生成
    if (!resource->CreateRenderTarget(width, height))
    {
        return nullptr;
    }

    // リソースをテーブルに追加
    const std::scoped_lock lock(_tableMutex);
    _resources.emplace(key, resource);

    return resource;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      レンダーターゲットの生成
 *  \param[in]  width           幅
 *  \param[in]  height          高さ
 *  \return     生成されたテクスチャリソース．失敗時にはnullptr．
 */
/* ------------------------------------------------------------------------- */
SharedTextureResource TextureStorage::CreateRenderTarget(uint32_t width, uint32_t height) noexcept
{
    // 派生テクスチャリソースを生成
    auto resource = CreateDerivedResource();
    if (resource == nullptr)
    {
        return nullptr;
    }

    // レンダーターゲットを生成
    if (!resource->CreateRenderTarget(width, height))
    {
        return nullptr;
    }

    return resource;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャリソースの取得
 *  \param[in]  key     テクスチャリソースのキー
 *  \return     キーに対応したテクスチャリソース．見つからない場合はnullptr．
 */
/* ------------------------------------------------------------------------- */
SharedTextureResource TextureStorage::Get(TextureKey key) noexcept
{
    const std::scoped_lock lock(_tableMutex);

    if (auto it = _resources.find(key); it != _resources.end())
    {
        return it->second;
    }

    return nullptr;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャリソースを破棄
 *  \param[in]  key     破棄するテクスチャリソースのキー
 */
/* ------------------------------------------------------------------------- */
bool TextureStorage::Destroy(TextureKey key) noexcept
{
    const std::scoped_lock lock(_tableMutex);

    return _resources.erase(key) >= 1;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャローダーの登録
 *  \param[in]  loaderKey   ローダーキー
 *  \param[in]  loader      登録するローダー
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool TextureStorage::RegisterLoader(TextureLoaderKey loaderKey, UniqueTextureLoader loader) noexcept
{
    const std::scoped_lock lock(_tableMutex);

    // kDefaultLoaderKeyで定義されているキーは予約なので使用できない
    if (loaderKey == kDefaultLoaderKey)
    {
        return false;
    }

    // 既に登録されている場合は失敗
    if (_loaders.find(loaderKey) != _loaders.end())
    {
        return false;
    }

    _loaders.emplace(loaderKey, std::move(loader));

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャローダーを取得
 *  \param[in]  loaderKey  ローダーキー
 *  \return     loaderKeyに対応したローダー．取得できない場合はnullptr
 */
/* ------------------------------------------------------------------------- */
TextureLoader *TextureStorage::GetTextureLoader(TextureLoaderKey loaderKey) const noexcept
{
    // 引数にkDefaultLoaderKeyを指定した場合，SetDefaultLoader()で指定したキーで代替する
    if (loaderKey == kDefaultLoaderKey)
    {
        loaderKey = _defaultLoaderKey;
    }

    if (auto it = _loaders.find(loaderKey); it != _loaders.end())
    {
        return it->second.get();
    }

    return nullptr;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      派生テクスチャリソースを生成
 *  \return     生成されたリソースのポインタ
 */
/* ------------------------------------------------------------------------- */
SharedTextureResource TextureStorage::CreateDerivedResource() const noexcept
{
    if (_textureGenerator != nullptr)
    {
        return _textureGenerator->MakeTextureResource();
    }

    return nullptr;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief          リソースの読み込み中状態の設定
 *  \param[in,out]  resource    リソース
 *  \param[in]      isLoading   読み込み中フラグ
 */
/* ------------------------------------------------------------------------- */
void TextureStorage::SetLoading(TextureResource *resource, bool isLoading) noexcept
{
    if (resource == nullptr)
    {
        return;
    }

    resource->SetLoading(isLoading);

    if (isLoading)
    {
        ++_loadingCount;
    }
    else
    {
        --_loadingCount;
        _loadingCondition.notify_all();
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief  リソースの読み込みが完了するまで処理をブロックする
 */
/* ------------------------------------------------------------------------- */
void TextureStorage::WaitLoading() noexcept
{
    if (_loadingCount != 0)
    {
        std::unique_lock lock(_loadingMutex);
        _loadingCondition.wait(lock, [this]
        {
            return _loadingCount == 0;
        });
    }
}

}    // namespace MGL::Render

// vim: et ts=4 sw=4 sts=4
