// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_texture.cc
 *  \brief      MGL テクスチャ
 *  \date       Since: December 20, 2020. 8:47:58 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/render/mgl_texture.h>

#include <utility>

namespace MGL::Render
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  resource        テクスチャリソース
 */
/* ------------------------------------------------------------------------- */
Texture::Texture(SharedTextureResource resource) noexcept
    : _resource(std::move(resource))
    , _withKey(false)
    , _key()
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ（取得のみ）
 *  \param[in]  key     テクスチャキー
 */
/* ------------------------------------------------------------------------- */
Texture::Texture(TextureKey key) noexcept
    : _resource(nullptr)
    , _withKey(true)
    , _key(key)
{
    Load(key);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ（ファイルから読み込み）
 *  \param[in]  key         テクスチャキー
 *  \param[in]  imagePath   読み込むファイルのパス
 *  \param[in]  isAsync     非同期読み込みフラグ
 *  \param[in]  loaderKey   テクスチャローダーのキー
 */
/* ------------------------------------------------------------------------- */
Texture::Texture(TextureKey key, const File::PathView &imagePath, bool isAsync, TextureLoaderKey loaderKey) noexcept
    : _resource(nullptr)
    , _withKey(true)
    , _key(key)
{
    if (isAsync)
    {
        LoadAsync(key, imagePath, loaderKey);
    }
    else
    {
        Load(key, imagePath, loaderKey);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ（メモリから読み込み）
 *  \param[in]  key         テクスチャキー
 *  \param[in]  imageData   読み込むデータのアドレス
 *  \param[in]  dataSize    読み込むデータのサイズ
 *  \param[in]  isAsync     非同期読み込みフラグ
 *  \param[in]  loaderKey   テクスチャローダーのキー
 */
/* ------------------------------------------------------------------------- */
Texture::Texture(TextureKey key, const void *imageData, size_t dataSize, bool isAsync, TextureLoaderKey loaderKey) noexcept
    : _resource(nullptr)
    , _withKey(true)
    , _key(key)
{
    if (isAsync)
    {
        LoadAsync(key, imageData, dataSize, loaderKey);
    }
    else
    {
        Load(key, imageData, dataSize, loaderKey);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ（ファイルから読み込み）
 *  \param[in]  imagePath   読み込むファイルのパス
 *  \param[in]  isAsync     非同期読み込みフラグ
 *  \param[in]  loaderKey   テクスチャローダーのキー
 */
/* ------------------------------------------------------------------------- */
Texture::Texture(const File::PathView &imagePath, bool isAsync, TextureLoaderKey loaderKey) noexcept
    : _resource(nullptr)
    , _withKey(false)
    , _key()
{
    if (isAsync)
    {
        LoadAsync(imagePath, loaderKey);
    }
    else
    {
        Load(imagePath, loaderKey);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ（メモリから読み込み）
 *  \param[in]  imageData   読み込むデータのアドレス
 *  \param[in]  dataSize    読み込むデータのサイズ
 *  \param[in]  isAsync     非同期読み込みフラグ
 *  \param[in]  loaderKey   テクスチャローダーのキー
 */
/* ------------------------------------------------------------------------- */
Texture::Texture(const void *imageData, size_t dataSize, bool isAsync, TextureLoaderKey loaderKey) noexcept
    : _resource(nullptr)
    , _withKey(false)
    , _key()
{
    if (isAsync)
    {
        LoadAsync(imageData, dataSize, loaderKey);
    }
    else
    {
        Load(imageData, dataSize, loaderKey);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャをテクスチャストレージから読み込み
 *  \param[in]  key     テクスチャキー
 *  \retval     true    キーに対応したテクスチャが見つかった
 *  \retval     false   キーに対応したテクスチャが見つからなかった
 */
/* ------------------------------------------------------------------------- */
bool Texture::Load(TextureKey key) noexcept
{
    auto &storage = RendererSet::GetInstance().GetTextureStorage();
    _resource = storage.Get(key);
    _withKey = true;
    _key = key;

    return (_resource != nullptr);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャをファイルから読み込み
 *  \param[in]  key         テクスチャキー
 *  \param[in]  imagePath   画像ファイルのパス
 *  \param[in]  loaderKey   テクスチャローダーのキー
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool Texture::Load(TextureKey key, const File::PathView &imagePath, TextureLoaderKey loaderKey) noexcept
{
    auto &storage = RendererSet::GetInstance().GetTextureStorage();
    _resource = storage.Load(key, imagePath, loaderKey);
    _withKey = true;
    _key = key;

    return _resource != nullptr;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャをファイルから非同期で読み込み
 *  \param[in]  key         テクスチャキー
 *  \param[in]  imagePath   画像ファイルのパス
 *  \param[in]  loaderKey   テクスチャローダーのキー
 *  \retval     true        成功
 *  \retval     false       失敗
 *  \note
 *      非同期読み込みの場合は常にtrueが返るため，正常に読み込めたかどうかは
 *      IsLoading()とIsValid()で確認する．
 */
/* ------------------------------------------------------------------------- */
bool Texture::LoadAsync(TextureKey key, const File::PathView &imagePath, TextureLoaderKey loaderKey) noexcept
{
    auto &storage = RendererSet::GetInstance().GetTextureStorage();
    _resource = storage.LoadAsync(key, imagePath, loaderKey);
    _withKey = true;
    _key = key;

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャをメモリから読み込み
 *  \param[in]  key         テクスチャキー
 *  \param[in]  imageData   読み込むデータのアドレス
 *  \param[in]  dataSize    読み込むデータのサイズ
 *  \param[in]  loaderKey   テクスチャローダーのキー
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool Texture::Load(TextureKey key, const void *imageData, size_t dataSize, TextureLoaderKey loaderKey) noexcept
{
    auto &storage = RendererSet::GetInstance().GetTextureStorage();
    _resource = storage.Load(key, imageData, dataSize, loaderKey);
    _withKey = true;
    _key = key;

    return _resource != nullptr;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャをメモリから非同期で読み込み
 *  \param[in]  key         テクスチャキー
 *  \param[in]  imageData   読み込むデータのアドレス
 *  \param[in]  dataSize    読み込むデータのサイズ
 *  \param[in]  loaderKey   テクスチャローダーのキー
 *  \retval     true        成功
 *  \retval     false       失敗
 *  \note
 *      非同期読み込みの場合は常にtrueが返るため，正常に読み込めたかどうかは
 *      IsLoading()とIsValid()で確認する．
 */
/* ------------------------------------------------------------------------- */
bool Texture::LoadAsync(TextureKey key, const void *imageData, size_t dataSize, TextureLoaderKey loaderKey) noexcept
{
    auto &storage = RendererSet::GetInstance().GetTextureStorage();
    _resource = storage.LoadAsync(key, imageData, dataSize, loaderKey);
    _withKey = true;
    _key = key;

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャをファイルから読み込み
 *  \param[in]  imagePath   画像ファイルのパス
 *  \param[in]  loaderKey   テクスチャローダーのキー
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool Texture::Load(const File::PathView &imagePath, TextureLoaderKey loaderKey) noexcept
{
    auto &storage = RendererSet::GetInstance().GetTextureStorage();
    _resource = storage.Load(imagePath, loaderKey);
    _withKey = false;
    _key = TextureKey{0};

    return _resource != nullptr;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャをファイルから非同期で読み込み
 *  \param[in]  imagePath   画像ファイルのパス
 *  \param[in]  loaderKey   テクスチャローダーのキー
 *  \retval     true        成功
 *  \retval     false       失敗
 *  \note
 *      非同期読み込みの場合は常にtrueが返るため，正常に読み込めたかどうかは
 *      IsLoading()とIsValid()で確認する．
 */
/* ------------------------------------------------------------------------- */
bool Texture::LoadAsync(const File::PathView &imagePath, TextureLoaderKey loaderKey) noexcept
{
    auto &storage = RendererSet::GetInstance().GetTextureStorage();
    _resource = storage.LoadAsync(imagePath, loaderKey);
    _withKey = false;
    _key = TextureKey{0};

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャをメモリから読み込み
 *  \param[in]  imageData   読み込むデータのアドレス
 *  \param[in]  dataSize    読み込むデータのサイズ
 *  \param[in]  loaderKey   テクスチャローダーのキー
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool Texture::Load(const void *imageData, size_t dataSize, TextureLoaderKey loaderKey) noexcept
{
    auto &storage = RendererSet::GetInstance().GetTextureStorage();
    _resource = storage.Load(imageData, dataSize, loaderKey);
    _withKey = false;
    _key = TextureKey{0};

    return _resource != nullptr;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャをメモリから非同期で読み込み
 *  \param[in]  imageData   読み込むデータのアドレス
 *  \param[in]  dataSize    読み込むデータのサイズ
 *  \param[in]  loaderKey   テクスチャローダーのキー
 *  \retval     true        成功
 *  \retval     false       失敗
 *  \note
 *      非同期読み込みの場合は常にtrueが返るため，正常に読み込めたかどうかは
 *      IsLoading()とIsValid()で確認する．
 */
/* ------------------------------------------------------------------------- */
bool Texture::LoadAsync(const void *imageData, size_t dataSize, TextureLoaderKey loaderKey) noexcept
{
    auto &storage = RendererSet::GetInstance().GetTextureStorage();
    _resource = storage.LoadAsync(imageData, dataSize, loaderKey);
    _withKey = false;
    _key = TextureKey{0};

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャをピクセルデータから生成
 *  \param[in]  key         テクスチャキー
 *  \param[in]  pixelData   ピクセルデータ
 *  \param[in]  pixelFormat ピクセルデータのフォーマット
 *  \param[in]  width       幅
 *  \param[in]  height      高さ
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool Texture::Create(TextureKey key, const void *pixelData, PixelFormat pixelFormat, uint32_t width, uint32_t height) noexcept
{
    auto &storage = RendererSet::GetInstance().GetTextureStorage();
    _resource = storage.Create(key, pixelData, pixelFormat, width, height);
    _withKey = true;
    _key = key;

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャをピクセルデータから生成
 *  \param[in]  pixelData   ピクセルデータ
 *  \param[in]  pixelFormat ピクセルデータのフォーマット
 *  \param[in]  width       幅
 *  \param[in]  height      高さ
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool Texture::Create(const void *pixelData, PixelFormat pixelFormat, uint32_t width, uint32_t height) noexcept
{
    auto &storage = RendererSet::GetInstance().GetTextureStorage();
    _resource = storage.Create(pixelData, pixelFormat, width, height);
    _withKey = false;
    _key = TextureKey{0};

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      レンダーターゲットの生成
 *  \param[in]  key         テクスチャキー
 *  \param[in]  width       幅
 *  \param[in]  height      高さ
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool Texture::CreateRenderTarget(TextureKey key, uint32_t width, uint32_t height) noexcept
{
    if (_resource != nullptr)
    {
        return false;
    }

    auto &storage = RendererSet::GetInstance().GetTextureStorage();

    _resource = storage.CreateRenderTarget(key, width, height);
    _withKey = true;
    _key = key;

    return (_resource != nullptr);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      レンダーターゲットの生成
 *  \param[in]  width       幅
 *  \param[in]  height      高さ
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool Texture::CreateRenderTarget(uint32_t width, uint32_t height) noexcept
{
    if (_resource != nullptr)
    {
        return false;
    }

    auto &storage = RendererSet::GetInstance().GetTextureStorage();

    _resource = storage.CreateRenderTarget(width, height);
    _withKey = false;

    return (_resource != nullptr);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャリソースの取得
 *  \return     テクスチャリソース
 */
/* ------------------------------------------------------------------------- */
SharedTextureResource Texture::GetResource() const noexcept
{
    if (!IsValid() || IsLoading())
    {
        return nullptr;
    }

    return _resource;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャが有効かを取得
 *  \retval     true    有効
 *  \retval     false   無効
 */
/* ------------------------------------------------------------------------- */
bool Texture::IsValid() const noexcept
{
    if (_resource == nullptr)
    {
        return false;
    }

    return _resource->IsValid();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャが読み込み中かを取得
 *  \retval     true    読み込み中
 *  \retval     false   読み込み中でない
 */
/* ------------------------------------------------------------------------- */
bool Texture::IsLoading() const noexcept
{
    if (_resource == nullptr)
    {
        return false;
    }

    return _resource->IsLoading();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャのサイズを取得
 *  \return     テクスチャのサイズ
 */
/* ------------------------------------------------------------------------- */
Vector2 Texture::GetSize() const noexcept
{
    if (!IsValid())
    {
        return {0.0f, 0.0f};
    }

    return _resource->GetSize();
}

}    // namespace MGL::Render

// vim: et ts=4 sw=4 sts=4
