// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       sprite_shader.metal
 *  \brief      スプライト用シェーダ
 *  \date       Since: November 12, 2020. 20:26:36 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <metal_stdlib>
#include <mgl/render/metal/shader/mgl_metal_shader_types.h>

//! ラスタライザデータ
struct RasterizerData
{
    float4 position [[position]];   //!< 位置
    float4 color;                   //!< 色
    float2 textureCoord;            //!< テクスチャ座標
};


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      矩形用頂点シェーダ
 *  \param[in]  vertexID            頂点ID
 *  \param[in]  instanceID          インスタンスID
 *  \param[in]  vertices            頂点情報
 *  \param[in]  attributes          アトリビュート
 *  \param[in]  projectionMatrix    投影変換行列
 *  \return     ラスタライズ後の頂点
 */
/* ------------------------------------------------------------------------- */
vertex RasterizerData spriteVertexShader(
            uint vertexID [[vertex_id]],
            uint instanceID [[instance_id]],
            constant PlaneModelVertex *vertices [[buffer(kSpriteShaderInputVertices)]],
            constant SpriteAttribute *attributes [[buffer(kSpriteShaderInputAttribute)]],
            constant metal::float4x4 &projectionMatrix [[buffer(kSpriteShaderInputProjectionMatrix)]])
{
    RasterizerData out;

    // 移動と回転の行列を生成
    constant SpriteAttribute *attribute = &attributes[instanceID];
    metal::float4x4 trMatrix;
    float cosZ = metal::cos(attribute->rotate);
    float sinZ = metal::sin(attribute->rotate);
    trMatrix[0] = float4(cosZ * attribute->flipX, sinZ * attribute->flipY, 0.0f, 0.0f);
    trMatrix[1] = float4(-sinZ * attribute->flipX, cosZ * attribute->flipY, 0.0f, 0.0f);
    trMatrix[2] = float4(0.0f, 0.0f, attribute->flipX * attribute->flipY, 0.0f);
    trMatrix[3] = float4(attribute->offset.x, attribute->offset.y, 0.0f, 1.0f);

    // 頂点と行列から平面上の座標を計算
    constant PlaneModelVertex *inVertex = &vertices[vertexID];
    out.position = projectionMatrix * trMatrix * float4(
                                                        (inVertex->position.x - attribute->pivot.x) * attribute->scale.x,
                                                        (inVertex->position.y - attribute->pivot.y) * attribute->scale.y,
                                                        0,
                                                        1);

    // 色はアトリビュートの色をそのまま出力
    out.color = attribute->color;
    
    // テクスチャ座標を指定
    out.textureCoord[0] = attribute->textureCoord[inVertex->textureCoordIndex[0]];
    out.textureCoord[1] = attribute->textureCoord[inVertex->textureCoordIndex[1]];
    
    return out;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      スプライト用フラグメントシェーダ
 *  \param[in]  in      ラスタライズ後の頂点情報
 *  \param[in]  texture テクスチャ
 *  \param[in]  sampler サンプラー
 *  \return     出力する色
 */
/* ------------------------------------------------------------------------- */
fragment float4 spriteFragmentShader(
            RasterizerData in [[stage_in]],
            metal::texture2d<float> texture [[texture(0)]],
            metal::sampler sampler [[sampler(0)]])
{
    return texture.sample(sampler, in.textureCoord) * in.color;
}

// vim: et ts=4 sw=4 sts=4 ft=cpp
