// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       rectangle_shader.metal
 *  \brief      矩形用シェーダ
 *  \date       Since: November 5, 2020. 17:05:59 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <metal_stdlib>
#include <mgl/render/metal/shader/mgl_metal_shader_types.h>

//! ラスタライザデータ
struct RasterizerData
{
    float4 position [[position]];   //!< 位置
    float4 color;                   //!< 色
};


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      矩形用頂点シェーダ
 *  \param[in]  vertexID            頂点ID
 *  \param[in]  instanceID          インスタンスID
 *  \param[in]  vertices            頂点情報
 *  \param[in]  attributes          アトリビュート
 *  \param[in]  projectionMatrix    投影変換行列
 *  \return     ラスタライズ後の頂点
 */
/* ------------------------------------------------------------------------- */
vertex RasterizerData rectangleVertexShader(
            uint vertexID [[vertex_id]],
            uint instanceID [[instance_id]],
            constant PlaneModelVertex *vertices [[buffer(kRectangleShaderInputVertices)]],
            constant RectangleAttribute *attributes [[buffer(kRectangleShaderInputAttribute)]],
            constant metal::float4x4 &projectionMatrix [[buffer(kRectangleShaderInputProjectionMatrix)]])
{
    RasterizerData out;

    // 移動と回転の行列を生成
    constant RectangleAttribute *attribute = &attributes[instanceID];
    metal::float4x4 trMatrix;
    trMatrix[0] = float4(metal::cos(attribute->rotate), metal::sin(attribute->rotate), 0.0f, 0.0f);
    trMatrix[1] = float4(metal::sin(-attribute->rotate), metal::cos(attribute->rotate), 0.0f, 0.0f);
    trMatrix[2] = float4(0.0f, 0.0f, 1.0f, 0.0f);
    trMatrix[3] = float4(attribute->offset.x, attribute->offset.y, 0.0f, 1.0f);
    
    // 頂点と行列から平面上の座標を計算
    constant PlaneModelVertex *inVertex = &vertices[vertexID];
    out.position = projectionMatrix * trMatrix * float4(
                                                        (inVertex->position.x - attribute->pivot.x) * attribute->scale.x,
                                                        (inVertex->position.y - attribute->pivot.y) * attribute->scale.y,
                                                        0,
                                                        1);

    // 色はアトリビュートの色をそのまま出力
    out.color = attribute->color;
    
    return out;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      矩形用フラグメントシェーダ
 *  \param[in]  in      ラスタライズ後の頂点情報
 *  \return     出力する色
 */
/* ------------------------------------------------------------------------- */
fragment float4 rectangleFragmentShader(RasterizerData in [[stage_in]])
{
    return in.color;
}

// vim: et ts=4 sw=4 sts=4 ft=cpp
