// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_metalkit_texture_loader.mm
 *  \brief      MGL MetalKitによるテクスチャローダー
 *  \date       Since: December 12, 2020. 22:22:43 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/render/metal/mgl_metalkit_texture_loader.h>

#if defined(MGL_RENDERER_ENABLE_METAL)

#include <mgl/render/metal/mgl_metal_defs.h>
#include <mgl/render/metal/mgl_metal_renderer.h>
#include <mgl/render/metal/mgl_metal_texture_resource.h>
#include <mgl/file/mgl_file.h>
#include <mgl/system/mgl_system_debug_macro.h>

namespace MGL::Render::Metal
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルから読み込み
 *  \param[out] textureResource     読み込み先のテクスチャリソース
 *  \param[in]  path                ファイルのパス
 *  \retval     true                成功
 *  \retval     false               失敗
 */
/* ------------------------------------------------------------------------- */
bool MetalKitTextureLoader::Load(MGL::Render::TextureResource &textureResource, const File::PathView &path) noexcept
{
    // Metal用のリソース以外は受け付けない
    if (textureResource.GetRendererType() != kRendererTypeMetal)
    {
        return false;
    }
    
    try
    {
        File::ThrowingHandle file;
        file.Open(path);
        auto size = file.GetSize();
        auto buffer = STL::make_unique<std::byte[]>(size);
        size = file.Read(buffer.get(), size);
        return Load(textureResource, buffer.get(), size);
    }
    catch (const File::Exception &fileException)
    {
        MGL_ERROR(fileException.what());
        return false;
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      メモリ上のバッファから読み込み
 *  \param[out] textureResource     読み込み先のテクスチャリソース
 *  \param[in]  imageData           読み込むイメージデータ
 *  \param[in]  dataSize            読み込むイメージデータのサイズ
 *  \retval     true                成功
 *  \retval     false               失敗
 */
/* ------------------------------------------------------------------------- */
bool MetalKitTextureLoader::Load(MGL::Render::TextureResource &textureResource, const void *imageData, size_t dataSize) noexcept
{
    // Metal用のリソース以外は受け付けない
    if (textureResource.GetRendererType() != kRendererTypeMetal)
    {
        return false;
    }
    auto metalTextureResource = static_cast<MGL::Render::Metal::TextureResource*>(&textureResource);
    
    // MetalKitのローダーでテクスチャを生成
    metalTextureResource->Create([MGLMetalRenderer.sharedInstance loadTextureFromMemory:imageData
                                                                               dataSize:dataSize]);
    
    return textureResource.IsValid();
}
}   // namespace MGL::Render::Metal

#endif  // MGL_RENDER_ENABLE_METAL
// vim: et ts=4 sw=4 sts=4
