// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_metal_texture_resource.mm
 *  \brief      MGL Metal用テクスチャリソース
 *  \date       Since: December 8, 2020. 15:27:18 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/render/metal/mgl_metal_texture_resource.h>
#if defined(MGL_RENDERER_ENABLE_METAL)

#include <mgl/render/metal/mgl_metal_renderer.h>
#include <mgl/render/metal/mgl_metal_render_target.h>

namespace MGL::Render::Metal
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
TextureResource::TextureResource() noexcept
    : _derivedData(STL::make_unique<MetalTextureResource>())
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ（メインレンダーターゲット用）
 *  \param[in]  isMainRenderTarget       trueを指定した場合，このテクスチャリソースはメインのレンダーターゲットとして扱う
 */
/* ------------------------------------------------------------------------- */
TextureResource::TextureResource(bool isMainRenderTarget) noexcept
    : _derivedData(STL::make_unique<MetalTextureResource>())
{
    _derivedData->isMainRenderTarget = isMainRenderTarget;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デストラクタ
 */
/* ------------------------------------------------------------------------- */
TextureResource::~TextureResource() noexcept
{
    Destroy();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      有効状態の取得
 *  \retval     true    有効
 *  \retval     false   無効
 */
/* ------------------------------------------------------------------------- */
bool TextureResource::IsValid() const noexcept
{
    if ((_derivedData->texture != nil) || IsRenderTarget())
    {
        return true;
    }
    
    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      レンダーターゲットに設定可能かを取得
 *  \retval     true    設定可能
 *  \retval     false   不可能
 */
/* ------------------------------------------------------------------------- */
bool TextureResource::IsRenderTarget() const noexcept
{
    if (_derivedData->isMainRenderTarget)
    {
        return true;
    }
    
    if (_derivedData->renderTarget != nil)
    {
        return true;
    }
    
    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャの生成
 *  \param[in]  pixelData   ピクセルデータ
 *  \param[in]  pixelFormat ピクセルフォーマット
 *  \param[in]  width       幅
 *  \param[in]  height      高さ
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool TextureResource::Create(const void *pixelData, PixelFormat pixelFormat, uint32_t width, uint32_t height) noexcept
{
    // 既に生成されている場合は実行しない
    if (IsValid())
    {
        return false;
    }
    
    // MGL定義のピクセルフォーマットをMetalの定義に変換
    auto metalPixelFormat = GetMetalPixelFormat(pixelFormat);
    if (metalPixelFormat == MTLPixelFormatInvalid)
    {
        return false;
    }
    
    // テクスチャの生成
    _derivedData->texture = [MGLMetalRenderer.sharedInstance createTexture:pixelData
                                                               pixelFormat:metalPixelFormat
                                                             pixelPerBytes:MGL::Render::GetPixelPerBytes(pixelFormat)
                                                                      size:simd_make_uint2(width, height)];
    
    return (_derivedData->texture != nil);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      レンダーターゲットの生成
 *  \param[in]  width       幅
 *  \param[in]  height      高さ
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool TextureResource::CreateRenderTarget(uint32_t width, uint32_t height) noexcept
{
    // 既に生成されている場合は実行しない
    if (IsValid())
    {
        return false;
    }

    // レンダーターゲットの生成
    _derivedData->renderTarget = [MGLMetalRenderer.sharedInstance createRenderTarget:simd_make_uint2(width, height)];
    
    return (_derivedData->renderTarget != nil);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      リソースの破棄
 */
/* ------------------------------------------------------------------------- */
void TextureResource::Destroy() noexcept
{
    _derivedData->texture = nil;
    _derivedData->renderTarget = nil;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      Metal用テクスチャからテクスチャリソースを生成
 *  \param[in]  texture Metal用テクスチャ
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool TextureResource::Create(id<MTLTexture> texture) noexcept
{
    if (IsValid())
    {
        return false;
    }
    
    _derivedData->texture = texture;
    
    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      Metal用のレンダーターゲットを取得
 *  \return     Metal用のレンダーターゲット
 */
/* ------------------------------------------------------------------------- */
MGLMetalRenderTarget *TextureResource::GetRenderTarget() const noexcept
{
    if (!IsRenderTarget())
    {
        return nil;
    }
    
    if (_derivedData->isMainRenderTarget)
    {
        return MGLMetalRenderer.sharedInstance.mainRenderTarget;
    }
    
    return _derivedData->renderTarget;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      Metal用テクスチャの取得
 *  \return     Metal用テクスチャ
 */
/* ------------------------------------------------------------------------- */
id<MTLTexture> TextureResource::GetTexture() const noexcept
{
    if (IsRenderTarget())
    {
        if (_derivedData->isMainRenderTarget)
        {
            return MGLMetalRenderer.sharedInstance.mainRenderTarget.texture;
        }
        
        return _derivedData->renderTarget.texture;
    }
    
    return _derivedData->texture;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      Metal用のピクセルフォーマットの定義を取得
 *  \param[in]  pixelFormat     MGLで定義されたピクセルフォーマット
 *  \return     対応したMetal用のピクセルフォーマットの定義
 */
/* ------------------------------------------------------------------------- */
MTLPixelFormat TextureResource::GetMetalPixelFormat(PixelFormat pixelFormat) const noexcept
{
    switch (pixelFormat)
    {
        case PixelFormat::RGBA8_UNorm:
            return MTLPixelFormatRGBA8Unorm;
            
        case PixelFormat::BGRA8_UNorm:
            return MTLPixelFormatBGRA8Unorm;
            
        case PixelFormat::B5G6R5_UNorm:
            if (@available(macOS 11.0, *))
            {
                return MTLPixelFormatB5G6R5Unorm;
            }
            return MTLPixelFormatInvalid;
            
        case PixelFormat::A1BGR5_UNorm:
            if (@available(macOS 11.0, *))
            {
                return MTLPixelFormatA1BGR5Unorm;
            }
            return MTLPixelFormatInvalid;

        case PixelFormat::BGR5A1_UNorm:
            if (@available(macOS 11.0, *))
            {
                return MTLPixelFormatBGR5A1Unorm;
            }
            return MTLPixelFormatInvalid;

        case PixelFormat::ABGR4_UNorm:
            if (@available(macOS 11.0, *))
            {
                return MTLPixelFormatABGR4Unorm;
            }
            return MTLPixelFormatInvalid;

        case PixelFormat::A8_UNorm:
            return MTLPixelFormatA8Unorm;
            
        case PixelFormat::BGRA4_UNorm:
        case PixelFormat::Unknown:
        case PixelFormat::Invalid:
            return MTLPixelFormatInvalid;
    }
    
    return MTLPixelFormatInvalid;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャのサイズを取得
 *  \return     テクスチャのサイズ
 */
/* ------------------------------------------------------------------------- */
Vector2 TextureResource::GetSize() const noexcept
{
    auto texture = GetTexture();
    if (texture == nil)
    {
        return Vector2(0.0f, 0.0f);
    }
    
    return Vector2(texture.width, texture.height);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャのピクセルフォーマットを取得
 *  \return     テクスチャのピクセルフォーマット
 */
/* ------------------------------------------------------------------------- */
PixelFormat TextureResource::GetPixelFormat() const noexcept
{
    auto texture = GetTexture();
    if (texture != nil)
    {
        switch (texture.pixelFormat)
        {
            case MTLPixelFormatRGBA8Unorm:
                return PixelFormat::RGBA8_UNorm;
                
            case MTLPixelFormatBGRA8Unorm:
                return PixelFormat::BGRA8_UNorm;
            
            case MTLPixelFormatB5G6R5Unorm:
                return PixelFormat::B5G6R5_UNorm;
                
            case MTLPixelFormatA1BGR5Unorm:
                return PixelFormat::A1BGR5_UNorm;
                
            case MTLPixelFormatBGR5A1Unorm:
                return PixelFormat::BGR5A1_UNorm;
                
            case MTLPixelFormatABGR4Unorm:
                return PixelFormat::ABGR4_UNorm;
                                
            case MTLPixelFormatA8Unorm:
                return PixelFormat::A8_UNorm;
                
            default:
                break;
        }
    }
    
    return PixelFormat::Unknown;
}
}   // namespace MGL::Render::Metal
#endif  // MGL_RENDERER_ENABLE_METAL
// vim: et ts=4 sw=4 sts=4
