// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_metal_sprite_renderer.mm
 *  \brief      MGL Metal用スプライトレンダラ
 *  \date       Since: December 13, 2020. 0:44:01 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/render/metal/mgl_metal_sprite_renderer.h>

#if defined(MGL_RENDERER_ENABLE_METAL)
#include <mgl/render/metal/mgl_metal_renderer.h>
#include <mgl/render/metal/mgl_metal_texture_resource.h>
#include <mgl/render/metal/mgl_metal_defs.h>

namespace MGL::Render::Metal
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
SpriteRenderer::SpriteRenderer() noexcept
    : _attributes(nullptr)
    , _backupAttributesPtr(nullptr)
    , _attributeCount(0)
    , _currentTextureResource(nullptr)
    , _currentSampler(SamplerType::Invalid)
    , _isEnabledBlending(true)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      描画の実行
 */
/* ------------------------------------------------------------------------- */
void SpriteRenderer::Flush() noexcept
{
    // アトリビュートが追加されていなければ実行しない
    if (_attributeCount == 0)
    {
        return;
    }
    
    // 描画を実行
    if (_currentTextureResource->GetRendererType() == kRendererTypeMetal)
    {
        auto metalTextureResource = static_cast<const MGL::Render::Metal::TextureResource*>(_currentTextureResource.get());
        
        auto renderer = MGLMetalRenderer.sharedInstance;
        [renderer drawSprite:_attributes
              attributeCount:_attributeCount
                     texture:metalTextureResource->GetTexture()
                samplerState:[renderer getSamplerState:_currentSampler]
           isEnabledBlending:_isEnabledBlending];
    }
    
    // 描画後はパラメータを元に戻す
    _attributes = nullptr;
    _attributeCount = 0;
    _currentSampler = SamplerType::Invalid;
    _currentTextureResource = nullptr;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アトリビュートを追加
 *  \param[in]  attribute           追加する描画アトリビュート
 *  \param[in]  textureResource     描画するテクスチャリソース
 *  \param[in]  sampler             使用するサンプラー
 *  \param[in]  isEnabledBlending   ブレンド設定の有効フラグ
 */
/* ------------------------------------------------------------------------- */
void SpriteRenderer::AddAttribute(const SpriteAttribute &attribute, SharedTextureResource &textureResource, SamplerType sampler, bool isEnabledBlending) noexcept
{
    // 描画しようとしているテクスチャやサンプラーの種類が変更された際には一旦描画を実行する
    if ((textureResource != _currentTextureResource) ||
        (sampler != _currentSampler) ||
        (isEnabledBlending != _isEnabledBlending))
    {
        Flush();
    }
    
    // アトリビュートを格納するバッファを用意
    if (_attributes == nullptr)
    {
        _attributes = reinterpret_cast<SpriteAttribute *>([MGLMetalRenderer.sharedInstance reserveDynamicBuffer:sizeof(SpriteAttribute) * kSpriteAttributeMax]);
        
        // レンダラからバッファを予約できなかった場合は自前で用意する
        if (_attributes == nil)
        {
            if (_backupAttributesPtr == nullptr)
            {
                _backupAttributesPtr = STL::make_unique<SpriteAttribute[]>(kSpriteAttributeMax);
            }
            _attributes = _backupAttributesPtr.get();
        }
    }
    
    // アトリビュートをテーブルに追加
    _attributes[_attributeCount] = attribute;
    if (++_attributeCount >= kSpriteAttributeMax)
    {
        Flush();
    }
    
    // 描画しようとしているテクスチャとサンプラーを保持しておく
    _currentTextureResource = textureResource;
    _currentSampler = sampler;
    _isEnabledBlending = isEnabledBlending;
}
}   // MGL::Render::Metal

#endif  // MGL_RENDERER_ENABLE_METAL
// vim: et ts=4 sw=4 sts=4
