// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_metal_renderer_2d.mm
 *  \brief      MGL Metal 2Dレンダラ
 *  \date       Since: November 29, 2020. 12:03:27 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/render/metal/mgl_metal_renderer_2d.h>
#if defined(MGL_RENDERER_ENABLE_METAL)

#include <mgl/math/mgl_math_defs.h>
#include <mgl/render/metal/mgl_metal_renderer.h>

namespace MGL::Render::Metal
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief  コンストラクタ
 */
/* ------------------------------------------------------------------------- */
Renderer2D::Renderer2D() noexcept
    : _eventPostFrameUpdate(Event::NotifyType::PostFrameUpdate, CallbackPostFrameUpdate, this)
    , _currentObjectRenderer(nullptr)
    , _lineRenderer()
    , _rectangleRenderer()
    , _spriteRenderer()
    , _mainRenderTarget(STL::make_shared<TextureResource>(true))
    , _currentRenderTarget(_mainRenderTarget)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief  レンダラが初期化可能な状態かを取得
 *  \retval true    初期化可能
 *  \retval false   初期化不可能
 */
/* ------------------------------------------------------------------------- */
bool Renderer2D::CanInitialize() const noexcept
{
    return (MGLMetalRenderer.sharedInstance != nullptr);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief  初期化処理
 *  \retval true    成功
 *  \retval false   失敗
 */
/* ------------------------------------------------------------------------- */
bool Renderer2D::Initialize() noexcept
{
    if (!CanInitialize())
    {
        return false;
    }
    
    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief  描画の実行
 */
/* ------------------------------------------------------------------------- */
void Renderer2D::Flush() noexcept
{
    ChangeObjectRenderer(nullptr);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      描画領域のクリア
 *  \param[in]  color   クリアする色
 */
/* ------------------------------------------------------------------------- */
void Renderer2D::Clear(const Color &color) noexcept
{
    Flush();
    [MGLMetalRenderer.sharedInstance clear:MTLClearColorMake(color.red, color.green, color.blue, color.alpha)];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ラインの描画
 *  \param[in]  start   開始点
 *  \param[in]  end     終了点
 *  \param[in]  color   描画する色
 */
/* ------------------------------------------------------------------------- */
void Renderer2D::DrawLine(const Vector2 &start, const Vector2 &end, const Color &color) noexcept
{
    // 使用するオブジェクトレンダラを変更
    ChangeObjectRenderer(&_lineRenderer);
    
    // アトリビュートを追加
    _lineRenderer.Add(start, end, color);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      矩形の描画
 *  \param[in]  rectangle   描画する矩形
 *  \param[in]  color       描画する色
 *  \param[in]  option      描画オプション
 */
/* ------------------------------------------------------------------------- */
void Renderer2D::DrawRectangle(const Rectangle &rectangle, const Color &color, const DrawOption2D &option) noexcept
{
    // 使用するオブジェクトレンダラを変更
    ChangeObjectRenderer(&_rectangleRenderer);
    
    RectangleAttribute attribute;
    
    // アトリビュートのオフセット（矩形の表示位置）を設定
    auto offset = option.GetOffset();
    attribute.offset.x = rectangle.x + offset.x;
    attribute.offset.y = rectangle.y + offset.y;
    AdjustAlignment(attribute.offset, rectangle.GetSize(), option);
    
    // アトリビュートのスケール（矩形のサイズ）を設定
    auto scale = option.GetScale();
    attribute.scale.x = rectangle.width * scale.x;
    attribute.scale.y = rectangle.height * scale.y;
    
    // 回転角度を設定
    attribute.rotate = option.GetRotate() * Math::kDegreeToRadian;
    attribute.pivot.x = option.GetPivot().x;
    attribute.pivot.y = option.GetPivot().y;

    // 色を設定
    auto maskColor = option.GetMaskColor();
    attribute.color[0] = color.red * maskColor.red;
    attribute.color[1] = color.green * maskColor.green;
    attribute.color[2] = color.blue * maskColor.blue;
    attribute.color[3] = color.alpha * maskColor.alpha;
    
    // アトリビュートを追加
    _rectangleRenderer.AddAttribute(attribute);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      スプライトの描画
 *  \param[in]  textureResource     描画するテクスチャのリソース
 *  \param[in]  position            描画する位置
 *  \param[in]  sourceRectangle     描画するテクスチャの領域
 *  \param[in]  option              描画オプション
 */
/* ------------------------------------------------------------------------- */
void Renderer2D::DrawSprite(
        const Vector2 &position,
        SharedTextureResource textureResource,
        const Rectangle &sourceRectangle,
        const DrawOption2D &option) noexcept
{
    if (!textureResource->IsValid())
    {
        return;
    }
    
    // 使用するオブジェクトレンダラを変更
    ChangeObjectRenderer(&_spriteRenderer);
    
    SpriteAttribute attribute;
    
    // アトリビュートのオフセット（矩形の表示位置）を設定
    auto offset = option.GetOffset();
    attribute.offset.x = position.x + offset.x;
    attribute.offset.y = position.y + offset.y;
    AdjustAlignment(attribute.offset, sourceRectangle.GetSize(), option);

    // アトリビュートのスケール（矩形のサイズ）を設定
    auto scale = option.GetScale();
    attribute.scale.x = sourceRectangle.width * scale.x;
    attribute.scale.y = sourceRectangle.height * scale.y;
    
    // テクスチャ座標を設定
    auto textureSize = textureResource->GetSize();
    attribute.textureCoord[0] = sourceRectangle.x / textureSize.x;
    attribute.textureCoord[1] = sourceRectangle.y / textureSize.y;
    attribute.textureCoord[2] = attribute.textureCoord[0] + sourceRectangle.width / textureSize.x;
    attribute.textureCoord[3] = attribute.textureCoord[1] + sourceRectangle.height / textureSize.y;
    
    // 回転角度を設定
    attribute.rotate = option.GetRotate() * Math::kDegreeToRadian;
    attribute.flipX = option.IsHorizontalFlip() ? -1.0f : 1.0f;
    attribute.flipY = option.IsVerticalFlip() ? -1.0f: 1.0f;
    attribute.pivot.x = option.GetPivot().x;
    attribute.pivot.y = option.GetPivot().y;
    if (attribute.flipX != attribute.flipY)
    {
        attribute.rotate = -attribute.rotate;
    }

    // 色を設定
    auto maskColor = option.GetMaskColor();
    attribute.color = { maskColor.red, maskColor.green, maskColor.blue, maskColor.alpha };
    
    // アトリビュートを追加
    _spriteRenderer.AddAttribute(attribute, textureResource, option.GetSamplerType(), option.GetBlendMode() != BlendMode::None);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      レンダーターゲットの設定
 *  \param[in]  renderTarget    設定するレンダーターゲット
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool Renderer2D::SetRenderTarget(SharedTextureResource renderTarget) noexcept
{
    // レンダーターゲットでなければ失敗
    if (!renderTarget->IsRenderTarget())
    {
        return false;
    }
    
    // Metal用のリソースでなければ失敗
    if (renderTarget->GetRendererType() != kRendererTypeMetal)
    {
        return false;
    }
    
    // 現在と同じレンダーターゲットであれば何もしない
    if (renderTarget == _currentRenderTarget)
    {
        return true;
    }
    
    // 一旦描画を実行しておく
    Flush();
    
    // レンダーターゲットを設定
    auto metalRenderTarget = static_cast<const TextureResource*>(renderTarget.get());
    [MGLMetalRenderer.sharedInstance setRenderTarget:metalRenderTarget->GetRenderTarget()];
    _currentRenderTarget = renderTarget;

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      現在のレンダーターゲットを取得
 *  \return     現在のレンダーターゲット
 */
/* ------------------------------------------------------------------------- */
SharedTextureResource Renderer2D::GetRenderTarget() const noexcept
{
    return _currentRenderTarget;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      メインレンダーターゲットを取得
 *  \return     メインレンダーターゲット
 */
/* ------------------------------------------------------------------------- */
SharedTextureResource Renderer2D::GetMainRenderTarget() const noexcept
{
    return _mainRenderTarget;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      シザーを設定
 *  \param[in]  isEnabled   有効フラグ
 *  \param[in]  rectangle   設定する矩形
 */
/* ------------------------------------------------------------------------- */
void Renderer2D::SetScissor(bool isEnabled, const Rectangle &rectangle) noexcept
{
    auto renderer = MGLMetalRenderer.sharedInstance;
    
    bool isSameEnable = renderer.isEnabledScissor == isEnabled;

    // 設定済みのフラグと設定しようとしているフラグがどちらも無効であれば何もしない
    if (isSameEnable && !isEnabled)
    {
        return;
    }
    
    MTLScissorRect scissorRectangle =
    {
        static_cast<NSUInteger>(rectangle.x),
        static_cast<NSUInteger>(rectangle.y),
        static_cast<NSUInteger>(rectangle.width),
        static_cast<NSUInteger>(rectangle.height)
    };
    
    // 設定されているパラメータが同一であれば何もしない
    if (isSameEnable)
    {
        if ((renderer.scissorRectangle.x == scissorRectangle.x) &&
            (renderer.scissorRectangle.y == scissorRectangle.y) &&
            (renderer.scissorRectangle.width == scissorRectangle.width) &&
            (renderer.scissorRectangle.height == scissorRectangle.height))
        {
            return;
        }
    }

    // 変更前に一旦描画を実行する
    Flush();

    // レンダラに設定
    renderer.isEnabledScissor = isEnabled;
    if (isEnabled)
    {
        renderer.scissorRectangle = scissorRectangle;
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      シザーが有効かどうかを取得
 *  \retval     true    有効
 *  \retval     false   無効
 */
/* ------------------------------------------------------------------------- */
bool Renderer2D::IsEnabledScissor() const noexcept
{
    return MGLMetalRenderer.sharedInstance.isEnabledScissor;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      シザー矩形を取得
 *  \return     シザー矩形
 */
/* ------------------------------------------------------------------------- */
Rectangle Renderer2D::GetScissorRectangle() const noexcept
{
    auto scissorRect = MGLMetalRenderer.sharedInstance.scissorRectangle;

    Rectangle rectangle =
    {
        static_cast<float>(scissorRect.x),
        static_cast<float>(scissorRect.y),
        static_cast<float>(scissorRect.width),
        static_cast<float>(scissorRect.height)
    };

    return rectangle;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      オブジェクトレンダラの変更
 *  \param[in]  objectRenderer  新たに設定するオブジェクトレンダラ
 */
/* ------------------------------------------------------------------------- */
void Renderer2D::ChangeObjectRenderer(MGL::Render::ObjectRenderer *objectRenderer) noexcept
{
    if (objectRenderer != _currentObjectRenderer)
    {
        if (_currentObjectRenderer != nullptr)
        {
            _currentObjectRenderer->Flush();
        }
    }
    
    _currentObjectRenderer = objectRenderer;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          オフセットを位置情報で補正
 *  \param[in,out]  offset  反映させるオフセット
 *  \param[in]      size    矩形のサイズ
 *  \param[in]      option  描画オプション
 */
/* ------------------------------------------------------------------------- */
void Renderer2D::AdjustAlignment(vector_float2 &offset, const Vector2 &size, const DrawOption2D &option) const noexcept
{
    auto scale = option.GetScale();
    auto width = size.x * scale.x;
    auto height = size.y * scale.y;
    
    // 水平方向の補正
    switch (option.GetAlignment().horizontal)
    {
        case Alignment::Horizontal::Center:
            offset.x -= width * 0.5f;
            break;

        case Alignment::Horizontal::Right:
            offset.x -= width;
            break;

        default:
            break;
    }

    // 垂直方向の補正
    switch (option.GetAlignment().vertical)
    {
        case Alignment::Vertical::Middle:
            offset.y -= height * 0.5f;
            break;

        case Alignment::Vertical::Bottom:
            offset.y -= height;
            break;

        default:
            break;
    }

    // 回転ピボットの補正
    auto &pivot = option.GetPivot();
    offset.x += pivot.x * width;
    offset.y += pivot.y * height;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フレーム更新前の処理
 *  \param[in]  callbackArg     コールバック関数の引数（Renderer2Dクラスのアドレス）
 *  \param[in]  notifyArg       未使用
 */
/* ------------------------------------------------------------------------- */
void Renderer2D::CallbackPostFrameUpdate(void *callbackArg, [[maybe_unused]] void *notifyArg) noexcept
{
    auto *thisPtr = static_cast<Renderer2D *>(callbackArg);
    
    thisPtr->Flush();
    thisPtr->SetRenderTarget(thisPtr->GetMainRenderTarget());
}

}   // namespce MGL::Render::Metal

#endif  // MGL_RENDERER_ENABLE_METAL
// vim: et ts=4 sw=4 sts=4
