// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_metal_renderer.mm
 *  \brief      Metalレンダラ
 *  \date       Since: November 3, 2020. 16:44:44 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#import <mgl/render/metal/mgl_metal_renderer.h>
#import <mgl/render/metal/mgl_metal_render_target.h>
#import <mgl/render/metal/mgl_metal_dynamic_buffers.h>
#import <mgl/event/mgl_event.h>
#import <mgl/math/mgl_vector2.h>

static MGLMetalRenderer *s_SharedInstance = nil;

static const size_t kDynamicBufferCount = 3;                //!< 動的バッファの数
static const size_t kDynamicBufferSize = 8 * 1024 * 1024;   //!< 動的バッファの1つあたりのサイズ

//! Metal用レンダラクラス
@implementation MGLMetalRenderer
{
    id<MTLDevice>           _device;            //!< Metalデバイス
    id<MTLCommandQueue>     _commandQueue;      //!< コマンドキュー
    id<MTLCommandBuffer>    _commandBuffer;     //!< コマンドバッファ
    MTKView                 *_mtkView;          //!< MetalKitビュークラス
    id<MTLLibrary>          _defaultLibrary;    //!< デフォルトライブラリ
    
    MTLViewport     _viewport;                          //!< ビューポート
    matrix_float4x4 _orthogonalProjectionMatrix;        //!< 平行投影行列
    id<MTLBuffer>   _orthogonalProjectionMatrixBuffer;  //!< 平行投影行列のリソースバッファ
    bool            _isChangedDrawableSize;             //!< Drawableのサイズに変更があったかのフラグ
    
    //! パイプラインステートの種類
    enum
    {
        kPipelineStateLine,                 //!< ライン
        kPipelineStateRectangle,            //!< 矩形
        kPipelineStateSprite,               //!< スプライト
        kPipelineStateSpriteWithoutBlend,   //!< スプライト（アルファブレンドなし）
        kPipelineStateMax                   //!< 最大数
    };
    id<MTLRenderPipelineState>  _pipelineStateArray[kPipelineStateMax];     //!< パイプラインステート

    //! サンプラーステートの種類
    enum
    {
        kSamplerStateNearest,       //!< 最近傍補間
        kSamplerStateLinear,        //!< 線形補間
        kSamplerStateMax            //!< 最大数
    };
    id<MTLSamplerState> _samplerStateArray[kSamplerStateMax];      //!< サンプラーステート

    id<MTLBuffer> _planeModel;      //!< 平面モデル
    
    bool _isRequestedClear;         //!< 画面クリア要求フラグ
    MTLClearColor _clearColor;      //!< 画面クリアの色
    
    MTLPixelFormat _mainRenderTargetPixelFormat;    //!< メインのレンダーターゲットのピクセルフォーマット
    MGLMetalRenderTarget *_currentRenderTarget;     //!< 現在のレンダーターゲット
    
    MGLMetalDynamicBuffers *_dynamicBuffers;        //!< 動的バッファ
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      インスタンスの取得
 *  \return     レンダラのインスタンス
 */
/* ------------------------------------------------------------------------- */
+ (instancetype) sharedInstance
{
    return s_SharedInstance;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \param[in]  mtkView     MetalKitビュークラス
 *  \return     初期化したこのクラス
 */
/* ------------------------------------------------------------------------- */
- (nonnull instancetype)initWithMetalKitView:(nonnull MTKView *)mtkView
{
    self = [super init];
    if (self == nil)
    {
        return nil;
    }
    
    _device = mtkView.device;
    
    // コマンドキューを生成
    _commandQueue = [_device newCommandQueue];
            
    // デフォルトのライブラリを読み込み
    NSBundle *bundle = [NSBundle bundleForClass:[self class]];
    NSString *libPath = [bundle pathForResource:@"default" ofType:@"metallib"];
    NSError *error = nil;
    _defaultLibrary = [_device newLibraryWithFile:libPath error:&error];
    if (_defaultLibrary == nil)
    {
        NSLog(@"[Renderer] Failed to load default library: %@", error);
        return nil;
    }

    // パイプラインステートを初期化
    if (![self initPipelineState:mtkView])
    {
        NSLog(@"[Renderer] Failed to initialize pipeline state.");
        return nil;
    }
    
    // 平面モデルを初期化
    if (![self initPlaneModel])
    {
        NSLog(@"[Renderer] Failed to initialize plane model.");
        return nil;
    }

    // サンプラーステートを初期化
    if (![self initSamplerState])
    {
        NSLog(@"[Renderer] Failed to initialize sampler state.");
        return nil;
    }

    // メインのレンダーターゲットを初期化（フレーム更新時に再設定されるので初期化時の中身は空）
    _mainRenderTarget = [[MGLMetalRenderTarget alloc] init];

    // 動的バッファを初期化
    _dynamicBuffers = [[MGLMetalDynamicBuffers alloc] initWithBufferCount:kDynamicBufferCount
                                                                   length:kDynamicBufferSize
                                                                   device:_device];

    // 画面クリアを予約しておく
    [self clear:MTLClearColorMake(0.0f, 0.0f, 0.0f, 1.0f)];

    // シザー矩形を初期化
    _isEnabledScissor = false;
    _scissorRectangle = {};

    // その他の変数を初期化
    _isChangedDrawableSize = true;
    _currentRenderTarget = nil;
    _mainRenderTargetPixelFormat = mtkView.colorPixelFormat;
    s_SharedInstance = self;
    
    return self;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      パイプラインステートの初期化
 *  \param[in]  mtkView     MetalKitビュークラス
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
- (bool)initPipelineState:(nonnull MTKView *)mtkView
{
    // 初期化パラメータテーブル
    const struct InitializeParameter
    {
        NSString *label;            // ラベル
        NSString *vertexShader;     // 頂点シェーダの名前
        NSString *fragmentShader;   // フラグメントシェーダの名前
        BOOL _isEnabledBlending;    // アルファブレンドの有効フラグ
    }
    initializeParameterTable[kPipelineStateMax] =
    {
        {@"LinePipeline",        @"lineVertexShader",        @"lineFragmentShader",         YES},   // kPipelineStateLine
        {@"RectanglePipeline",   @"rectangleVertexShader",   @"rectangleFragmentShader",    YES},   // kPipelineStateRectangle
        {@"SpritePipeline",      @"spriteVertexShader",      @"spriteFragmentShader",       YES},   // kPipelineStateSprite
        {@"SpritePipeline",      @"spriteVertexShader",      @"spriteFragmentShader",       NO},    // kPipelineStateSpriteWithoutBlend
    };
    
    for (size_t i = 0; i < kPipelineStateMax; i++)
    {
        const InitializeParameter &initParam = initializeParameterTable[i];
        
        MTLRenderPipelineDescriptor *pipelineStateDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
        
        // ラベルの設定
        pipelineStateDescriptor.label = initParam.label;
        
        // シェーダの設定
        pipelineStateDescriptor.vertexFunction = [_defaultLibrary newFunctionWithName:initParam.vertexShader];
        pipelineStateDescriptor.fragmentFunction = [_defaultLibrary newFunctionWithName:initParam.fragmentShader];
        
        // ピクセルフォーマットの設定
        pipelineStateDescriptor.colorAttachments[0].pixelFormat = mtkView.colorPixelFormat;
        
        // アルファブレンドの設定
        pipelineStateDescriptor.colorAttachments[0].blendingEnabled = initParam._isEnabledBlending;
        if (initParam._isEnabledBlending)
        {
            pipelineStateDescriptor.colorAttachments[0].rgbBlendOperation           = MTLBlendOperationAdd;
            pipelineStateDescriptor.colorAttachments[0].alphaBlendOperation         = MTLBlendOperationAdd;
            pipelineStateDescriptor.colorAttachments[0].sourceRGBBlendFactor        = MTLBlendFactorSourceAlpha;
            pipelineStateDescriptor.colorAttachments[0].sourceAlphaBlendFactor      = MTLBlendFactorSourceAlpha;
            pipelineStateDescriptor.colorAttachments[0].destinationRGBBlendFactor   = MTLBlendFactorOneMinusSourceAlpha;
            pipelineStateDescriptor.colorAttachments[0].destinationAlphaBlendFactor = MTLBlendFactorOneMinusSourceAlpha;
        }
        
        NSError *error = NULL;
        _pipelineStateArray[i] = [_device newRenderPipelineStateWithDescriptor:pipelineStateDescriptor
                                                                         error:&error];
        if (!_pipelineStateArray[i])
        {
            NSLog(@"[Renderer] Failed to r rectangle pipeline state: error %@", error);
            return false;
        }
    }
    
    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      サンプラーステートの初期化
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
- (bool)initSamplerState
{
    // 初期化パラメータテーブル
    const struct InitializeParameter
    {
        NSString *label;                    // ラベル
        MTLSamplerMinMagFilter minFilter;   // 縮小時のフィルタ
        MTLSamplerMinMagFilter magFilter;   // 拡大時のフィルタ
    }
    initializeParameterTable[kSamplerStateMax] =
    {
        {@"NearestSampler", MTLSamplerMinMagFilterNearest, MTLSamplerMinMagFilterNearest},     // kSamplerStateNearest
        {@"LinearSampler",  MTLSamplerMinMagFilterLinear,  MTLSamplerMinMagFilterLinear},      // kSamplerStateLinear
    };
    
    for (size_t i = 0; i < kSamplerStateMax; i++)
    {
        const InitializeParameter &initParam = initializeParameterTable[i];
        
        MTLSamplerDescriptor *samplerStateDescriptor = [[MTLSamplerDescriptor alloc] init];
        samplerStateDescriptor.label     = initParam.label;
        samplerStateDescriptor.minFilter = initParam.minFilter;
        samplerStateDescriptor.magFilter = initParam.magFilter;
        
        _samplerStateArray[i] = [_device newSamplerStateWithDescriptor:samplerStateDescriptor];
        if (_samplerStateArray[i] == nil)
        {
            return false;
        }
    }
    
    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      プライベートバッファを生成
 *  \param[in]  data    リソースに設定するデータ
 *  \param[in]  length  設定するデータのサイズ
 *  \return     新たに生成したプライベートバッファ
 */
/* ------------------------------------------------------------------------- */
- (nullable id<MTLBuffer>)newPrivateBufferWithBytes:(nonnull const void *)data
                                             length:(size_t)length
{
    // 共有バッファを用意
    id<MTLBuffer> sharedBuffer = [_device newBufferWithBytes:data
                                                      length:length
                                                     options:MTLResourceStorageModeShared];

    // プライベートバッファを用意
    id <MTLBuffer> privateBuffer = [_device newBufferWithLength:length
                                                        options:MTLResourceStorageModePrivate];

    // 共有バッファからプライベートバッファへ転送するコマンドエンコーダを生成
    id <MTLCommandBuffer> commandBuffer = [_commandQueue commandBuffer];
    id <MTLBlitCommandEncoder> blitCommandEncoder = [commandBuffer blitCommandEncoder];
    [blitCommandEncoder copyFromBuffer:sharedBuffer
                          sourceOffset:0
                              toBuffer:privateBuffer
                     destinationOffset:0
                                  size:length];
    [blitCommandEncoder endEncoding];

    // コマンドを実行
    [commandBuffer commit];
    
    return privateBuffer;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      平面モデルの生成
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
- (bool)initPlaneModel
{
    static PlaneModelVertex planeVertex[] =
    {
        {{0.0f, 0.0f}, {0, 1}},
        {{1.0f, 0.0f}, {2, 1}},
        {{0.0f, 1.0f}, {0, 3}},
        {{1.0f, 1.0f}, {2, 3}}
    };
    
    _planeModel = [self newPrivateBufferWithBytes:planeVertex
                                           length:sizeof(planeVertex)];
    if (_planeModel == nil)
    {
        return false;
    }
    
    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルからテクスチャを読み込み
 *  \param[in]  filePath    読み込むファイルのパス
 *  \return     生成されたテクスチャ
 */
/* ------------------------------------------------------------------------- */
- (nullable id<MTLTexture>)loadTexture:(nonnull NSString *)filePath
{
    NSString *imagePath = [[NSBundle mainBundle] pathForResource:filePath
                                                          ofType:nil];
    if (imagePath == nil)
    {
        return nil;
    }
    
    NSURL *url = [NSURL fileURLWithPath:imagePath];

    MTKTextureLoader *loader = [[MTKTextureLoader alloc] initWithDevice: _device];
    NSDictionary* options = @{
        MTKTextureLoaderOptionSRGB: [[NSNumber alloc] initWithBool:NO]
    };
    NSError *error = nil;
    id<MTLTexture> texture = [loader newTextureWithContentsOfURL:url
                                                         options:options
                                                           error:&error];
    if(texture == nil)
    {
       NSLog(@"[Renderer] Failed to create the texture from %@: %@", url.absoluteString, error);
       return nil;
    }

    return texture;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      メモリ上のバッファからテクスチャを読み込み
 *  \param[in]  imageData   読み込むイメージデータ
 *  \param[in]  dataSize    読み込むイメージデータのサイズ
 *  \return     生成されたテクスチャ
 */
/* ------------------------------------------------------------------------- */
- (nullable id<MTLTexture>)loadTextureFromMemory:(nonnull const void *)imageData
                                        dataSize:(size_t)dataSize
{
    NSData *data = [NSData dataWithBytes:imageData
                                  length:dataSize];
    MTKTextureLoader *loader = [[MTKTextureLoader alloc] initWithDevice: _device];
    NSDictionary* options = @{
        MTKTextureLoaderOptionSRGB: [[NSNumber alloc] initWithBool:NO]
    };
    NSError *error = nil;
    id<MTLTexture> texture = [loader newTextureWithData:data
                                                options:options
                                                  error:&error];
    if(texture == nil)
    {
       NSLog(@"[Renderer] Failed to create the texture from %@: %@", imageData, error);
       return nil;
    }

    return texture;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャの生成
 *  \param[in]  pixelData       ピクセルデータのバッファ
 *  \param[in]  pixelFormat     ピクセルデータのフォーマット
 *  \param[in]  pixelPerBytes   1ピクセルあたりのバイト数
 *  \param[in]  size            テクスチャのサイズ
 *  \return     生成されたテクスチャ．失敗時にはnil．
 */
/* ------------------------------------------------------------------------- */
- (nullable id<MTLTexture>)createTexture:(nonnull const void *)pixelData
                             pixelFormat:(MTLPixelFormat)pixelFormat
                           pixelPerBytes:(size_t)pixelPerBytes
                                    size:(simd_uint2)size
{
    // テクスチャ生成の記述子を準備
    MTLTextureDescriptor *textureDescriptor = [[MTLTextureDescriptor alloc] init];
    textureDescriptor.pixelFormat = pixelFormat;
    textureDescriptor.width = size.x;
    textureDescriptor.height = size.y;
    
    // テクスチャを生成
    id<MTLTexture> texture = [_device newTextureWithDescriptor:textureDescriptor];
    if (texture == nil)
    {
        return nil;
    }
    
    // ピクセルデータをテクスチャにコピー
    MTLRegion region =
    {
        {0, 0, 0},
        {(uint32_t)size.x, (uint32_t)size.y, 0}
    };
    NSUInteger bytesPerRow = size.x * pixelPerBytes;
    [texture replaceRegion:region
               mipmapLevel:0
                 withBytes:pixelData
               bytesPerRow:bytesPerRow];
        
    return texture;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      レンダーターゲットの生成
 *  \param[in]  size        サイズ
 *  \return     生成したレンダーターゲット．失敗時はnil．
 */
/* ------------------------------------------------------------------------- */
- (nullable MGLMetalRenderTarget*)createRenderTarget:(simd_uint2)size
{
    return [[MGLMetalRenderTarget alloc] initWithSize:size
                                          pixelFormat:_mainRenderTargetPixelFormat
                                               device:_device
                                         commandQueue:_commandQueue];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      レンダーターゲットの設定
 *  \param[in]  renderTarget    設定するレンダーターゲット
 */
/* ------------------------------------------------------------------------- */
- (void)setRenderTarget:(nonnull MGLMetalRenderTarget *)renderTarget
{
    // 描画先切り替え時に前回予約した画面クリアが実行されていなければ即時行う
    if (_currentRenderTarget != renderTarget)
    {
        if (_isRequestedClear)
        {
            [self clearImmediately:_clearColor];
            _isRequestedClear = NO;
        }
    }
    
    _currentRenderTarget = renderTarget;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      MGLのサンプラータイプからサンプラーステートを取得
 *  \param[in]  sampler     MGLのサンプラータイプ
 *  \return     対応したサンプラーステート．失敗時はnil．
 */
/* ------------------------------------------------------------------------- */
- (nullable id<MTLSamplerState>)getSamplerState:(MGL::Render::SamplerType)sampler
{
    switch (sampler)
    {
        case MGL::Render::SamplerType::Nearest:
            return _samplerStateArray[kSamplerStateNearest];
            
        case MGL::Render::SamplerType::Linear:
            return _samplerStateArray[kSamplerStateLinear];

        default:
            break;
    }
    
    return nil;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      リサイズ時のコールバック処理
 *  \param[in]  view    MetalKitビュークラス
 *  \param[in]  size    リサイズ後のサイズ
 */
/* ------------------------------------------------------------------------- */
-        (void)mtkView:(nonnull MTKView *)view
drawableSizeWillChange:(CGSize)size
{
    // ビューポートを更新
    _viewport.originX = 0;
    _viewport.originY = 0;
    _viewport.width = size.width;
    _viewport.height = size.height;
    _viewport.znear = -1.0;
    _viewport.zfar = 1.0;
    
    // 平行投影行列を更新
    _orthogonalProjectionMatrix.columns[0] = simd_make_float4(2.0f / (float)_viewport.width, 0.0f, 0.0f, 0.0f);
    _orthogonalProjectionMatrix.columns[1] = simd_make_float4(0.0f, 2.0f / -(float)_viewport.height, 0.0f, 0.0f);
    _orthogonalProjectionMatrix.columns[2] = simd_make_float4(0.0f, 0.0f, 1.0f, 0.0f);
    _orthogonalProjectionMatrix.columns[3] = simd_make_float4(-1.0f, 1.0f, 0.0f, 1.0f);

    _isChangedDrawableSize = true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フレーム描画処理
 *  \param[in]  view    MetalKitビュークラス
 */
/* ------------------------------------------------------------------------- */
- (void)drawInMTKView:(nonnull MTKView *)view
{
    // Drawableのサイズが変更されていた際の処理
    if (_isChangedDrawableSize)
    {
        // 平行投影行列を作り直す
        _orthogonalProjectionMatrixBuffer = [self newPrivateBufferWithBytes:&_orthogonalProjectionMatrix
                                                                     length:sizeof(matrix_float4x4)];
        
        // クライアントサイズ変更のイベント通知
        MGL::Vector2 newSize(static_cast<float>(_viewport.width), static_cast<float>(_viewport.height));
        MGL::Event::Notify(MGL::Event::NotifyType::ChangeClientSize, &newSize);
        
        _isChangedDrawableSize = false;
    }
    
    // 描画の準備
    _commandBuffer = [_commandQueue commandBuffer];
    _commandBuffer.label = @"Command";
    _mtkView = view;
    [_mainRenderTarget updateWithMetalKitView:view
                                     viewport:&_viewport
                   orthogonalProjectionMatrix:_orthogonalProjectionMatrixBuffer];
    _currentRenderTarget = _mainRenderTarget;

    // 動的バッファの切り替え
    [_dynamicBuffers next];

    // 各種フレーム動作
    MGL::Event::Notify(MGL::Event::NotifyType::PreFrameUpdate);
    MGL::Event::Notify(MGL::Event::NotifyType::AppFrameUpdate);
    MGL::Event::Notify(MGL::Event::NotifyType::PostFrameUpdate);

    // 画面クリアが予約されたまま実行されていなければ実行する
    if (_isRequestedClear)
    {
        [self clearImmediately:_clearColor];
    }

    // 描画先の指定
    [_commandBuffer presentDrawable:view.currentDrawable];

    // 描画が完了したら動的バッファのセマフォにシグナルを送信する
    __weak dispatch_semaphore_t semaphore = _dynamicBuffers.semaphore;
    [_commandBuffer addCompletedHandler:^(id<MTLCommandBuffer> commandBuffer)
    {
        (void)commandBuffer;
        dispatch_semaphore_signal(semaphore);
    }];
    
    // 描画をコミット
    [_commandBuffer commit];
    
    _currentRenderTarget = nil;
    _mtkView = nil;
    _commandBuffer = nil;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      描画領域のクリア
 *  \param[in]  color   クリアする色
 */
/* ------------------------------------------------------------------------- */
- (void)clear:(MTLClearColor)color
{
    _clearColor = color;
    _isRequestedClear = true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      描画領域の即時クリア
 *  \param[in]  color   クリアする色
 */
/* ------------------------------------------------------------------------- */
- (void)clearImmediately:(MTLClearColor)color
{
    MTLRenderPassDescriptor *renderPassDescriptor = _currentRenderTarget.renderPassDescriptor;
    if (renderPassDescriptor != nil)
    {
        renderPassDescriptor.colorAttachments[0].loadAction = MTLLoadActionClear;
        renderPassDescriptor.colorAttachments[0].clearColor = color;
        
        id<MTLRenderCommandEncoder> renderEncoder = [_commandBuffer renderCommandEncoderWithDescriptor:renderPassDescriptor];
        renderEncoder.label = @"ClearEncoder";
        
        [renderEncoder endEncoding];
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          画面クリアをレンダーパス記述子に適用
 *  \param[in,out]  descriptor  適用先のレンダーパス記述子
 */
/* ------------------------------------------------------------------------- */
- (void)applyClear:(nonnull MTLRenderPassDescriptor *)descriptor
{
    if (_isRequestedClear)
    {
        descriptor.colorAttachments[0].loadAction = MTLLoadActionClear;
        descriptor.colorAttachments[0].clearColor = _clearColor;
        _isRequestedClear = NO;
    }
    else
    {
        descriptor.colorAttachments[0].loadAction = MTLLoadActionLoad;
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ラインの描画
 *  \param[in]  vertices    頂点バッファ
 *  \param[in]  vertexCount 頂点バッファの数
 */
/* ------------------------------------------------------------------------- */
- (void)drawLine:(nonnull const LineShaderVertex *)vertices
     vertexCount:(size_t)vertexCount
{
    // 頂点のバッファとオフセットを取得
    size_t vertexBufferOffset = 0;
    id<MTLBuffer> vertexBuffer = [self getBufferAndOffset:&vertexBufferOffset
                                                     data:vertices
                                                   length:sizeof(LineShaderVertex) * vertexCount];

    // レンダーパスの記述子を初期化
    MTLRenderPassDescriptor *renderPassDescriptor = _currentRenderTarget.renderPassDescriptor;
    if(renderPassDescriptor != nil)
    {
        // 画面クリアを適用
        [self applyClear:renderPassDescriptor];
        
        // レンダーエンコーダを作成
        id<MTLRenderCommandEncoder> renderEncoder = [_commandBuffer renderCommandEncoderWithDescriptor:renderPassDescriptor];
        renderEncoder.label = @"DrawLineEncoder";

        // ビューポートを設定
        [renderEncoder setViewport:_currentRenderTarget.viewport];

        // パイプラインステートを設定
        [renderEncoder setRenderPipelineState:_pipelineStateArray[kPipelineStateLine]];

        // 頂点情報を設定
        [renderEncoder setVertexBuffer:vertexBuffer
                                offset:vertexBufferOffset
                               atIndex:kLineShaderInputVertices];

        // 平行投影行列を設定
        [renderEncoder setVertexBuffer:_currentRenderTarget.orthogonalProjectionMatrix
                                offset:0
                               atIndex:kLineShaderInputProjectionMatrix];

        // シザー矩形を設定
        if (_isEnabledScissor)
        {
            [renderEncoder setScissorRect:_scissorRectangle];
        }
        
        // プリミティブの描画
        [renderEncoder drawPrimitives:MTLPrimitiveTypeLine
                          vertexStart:0
                          vertexCount:vertexCount];

        [renderEncoder endEncoding];
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      矩形の描画
 *  \param[in]  attributes  描画する矩形アトリビュート
 *  \param[in]  count       描画する矩形の数
 */
/* ------------------------------------------------------------------------- */
- (void)drawRectangle:(nonnull const RectangleAttribute *)attributes
       attributeCount:(size_t)count
{
    // アトリビュートのバッファとオフセットを取得
    size_t attributeBufferOffset = 0;
    id<MTLBuffer> attributeBuffer = [self getBufferAndOffset:&attributeBufferOffset
                                                        data:attributes
                                                      length:sizeof(RectangleAttribute) * count];

    // レンダーパスの記述子を初期化
    MTLRenderPassDescriptor *renderPassDescriptor = _currentRenderTarget.renderPassDescriptor;
    if(renderPassDescriptor != nil)
    {
        // 画面クリアを適用
        [self applyClear:renderPassDescriptor];

        // レンダーエンコーダを作成
        id<MTLRenderCommandEncoder> renderEncoder = [_commandBuffer renderCommandEncoderWithDescriptor:renderPassDescriptor];
        renderEncoder.label = @"DrawRectangleEncoder";

        // ビューポートを設定
        [renderEncoder setViewport:_currentRenderTarget.viewport];

        // パイプラインステートを設定
        [renderEncoder setRenderPipelineState:_pipelineStateArray[kPipelineStateRectangle]];

        // 頂点情報を設定
        [renderEncoder setVertexBuffer:_planeModel
                                offset:0
                               atIndex:kRectangleShaderInputVertices];

        // 矩形アトリビュートを設定
        [renderEncoder setVertexBuffer:attributeBuffer
                                offset:attributeBufferOffset
                               atIndex:kRectangleShaderInputAttribute];

        // 平行投影行列を設定
        [renderEncoder setVertexBuffer:_currentRenderTarget.orthogonalProjectionMatrix
                                offset:0
                               atIndex:kRectangleShaderInputProjectionMatrix];

        // シザー矩形を設定
        if (_isEnabledScissor)
        {
            [renderEncoder setScissorRect:_scissorRectangle];
        }
        
        // プリミティブの描画
        [renderEncoder drawPrimitives:MTLPrimitiveTypeTriangleStrip
                          vertexStart:0
                          vertexCount:4
                        instanceCount:count];

        [renderEncoder endEncoding];
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      スプライトの描画
 *  \param[in]  attributes          描画するスプライトアトリビュート
 *  \param[in]  count               描画するスプライトの数
 *  \param[in]  texture             描画に使用するテクスチャ
 *  \param[in]  samplerState        描画に使用するサンプラーステート
 *  \param[in]  isEnabledBlending   アルファブレンドの有効フラグ
 */
/* ------------------------------------------------------------------------- */
- (void)drawSprite:(nonnull const SpriteAttribute *)attributes
    attributeCount:(size_t)count
           texture:(nonnull const id<MTLTexture>)texture
      samplerState:(nonnull const id<MTLSamplerState>)samplerState
 isEnabledBlending:(bool)isEnabledBlending
{
    if ((texture == nil) || (samplerState == nil))
    {
        return;
    }

    // アトリビュートのバッファとオフセットを取得
    size_t attributeBufferOffset = 0;
    id<MTLBuffer> attributeBuffer = [self getBufferAndOffset:&attributeBufferOffset
                                                        data:attributes
                                                      length:sizeof(SpriteAttribute) * count];        
    
    // レンダーパスの記述子を初期化
    MTLRenderPassDescriptor *renderPassDescriptor = _currentRenderTarget.renderPassDescriptor;
    if(renderPassDescriptor != nil)
    {
        // 画面クリアを適用
        [self applyClear:renderPassDescriptor];

        // レンダーエンコーダを作成
        id<MTLRenderCommandEncoder> renderEncoder = [_commandBuffer renderCommandEncoderWithDescriptor:renderPassDescriptor];
        renderEncoder.label = @"DrawSpriteEncoder";

        // ビューポートを設定
        [renderEncoder setViewport:_currentRenderTarget.viewport];

        // パイプラインステートを設定
        [renderEncoder setRenderPipelineState:_pipelineStateArray[(isEnabledBlending) ? kPipelineStateSprite : kPipelineStateSpriteWithoutBlend]];

        // 頂点情報を設定
        [renderEncoder setVertexBuffer:_planeModel
                                offset:0
                               atIndex:kSpriteShaderInputVertices];

        // 矩形アトリビュートを設定
        [renderEncoder setVertexBuffer:attributeBuffer
                                offset:attributeBufferOffset
                               atIndex:kSpriteShaderInputAttribute];

        // 平行投影行列を設定
        [renderEncoder setVertexBuffer:_currentRenderTarget.orthogonalProjectionMatrix
                                offset:0
                               atIndex:kSpriteShaderInputProjectionMatrix];

        // テクスチャを設定
        [renderEncoder setFragmentTexture:texture
                                  atIndex:0];
        
        // サンプラーを設定
        [renderEncoder setFragmentSamplerState:samplerState
                                       atIndex:0];

        // シザー矩形を設定
        if (_isEnabledScissor)
        {
            [renderEncoder setScissorRect:_scissorRectangle];
        }

        // プリミティブの描画
        [renderEncoder drawPrimitives:MTLPrimitiveTypeTriangleStrip
                          vertexStart:0
                          vertexCount:4
                        instanceCount:count];

        [renderEncoder endEncoding];
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      動的バッファからアドレスを予約
 *  \param[in]  length  予約するサイズ
 *  \return     予約したアドレス．失敗時はnil
 */
/* ------------------------------------------------------------------------- */
- (nullable void *)reserveDynamicBuffer:(size_t)length
{
    return [_dynamicBuffers reserve:length];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アドレスからGPUに渡すバッファとオフセットを取得
 *  \param[out] offset  GPUに渡すオフセットの格納先
 *  \param[in]  data    取得元のアドレス
 *  \param[in]  length  データサイズ
 *  \return     GPUに渡すバッファ
 */
/* ------------------------------------------------------------------------- */
- (id<MTLBuffer>)getBufferAndOffset:(size_t *)offset
                               data:(const void *)data
                             length:(size_t)length
{
    id<MTLBuffer> buffer = nil;
    
    // 入力データのアドレスと動的バッファの予約アドレスが一致していれば予約済みの情報を返す
    if ([_dynamicBuffers isReservedAddress:data])
    {
        buffer = _dynamicBuffers.currentBuffer;
        *offset = _dynamicBuffers.currentOffset;
        [_dynamicBuffers commit:length];
    }
    // 予約済みのアドレスを使用していない場合，動的バッファからアロケートしてデータのコピーを試みる
    else
    {
        void *allocatedPointer = [_dynamicBuffers allocate:length];
        if (allocatedPointer != nil)
        {
            memcpy(allocatedPointer, data, length);
            buffer = _dynamicBuffers.currentBuffer;
            *offset = _dynamicBuffers.prevOffset;
        }
    }
    
    // 動的バッファが利用できなかった場合，新たにバッファを生成する
    // NOTE: この処理に到達してしまうのはパフォーマンス上好ましくないため，動的バッファのサイズを調整するか処理を見直す必要がある
    if (buffer == nil)
    {
        buffer = [_device newBufferWithBytes:data
                                      length:length
                                     options:MTLResourceStorageModeShared];
        *offset = 0;
    }
    
    return buffer;
}
@end

// vim: et ts=4 sw=4 sts=4
