// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_metal_render_target.mm
 *  \brief      Metal用レンダーターゲット
 *  \date       Since: November 14, 2020. 12:32:58 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#import <mgl/render/metal/mgl_metal_render_target.h>
#if defined(MGL_RENDERER_ENABLE_METAL)

//! Metal用レンダーターゲットクラス
@implementation MGLMetalRenderTarget
{
    MTKView                 *_mtkView;                  //!< MetalKitビュー
    MTLRenderPassDescriptor *_renderPassDescriptor;     //!< レンダーパスの記述子
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化
 *  \return     自身のクラス
 */
/* ------------------------------------------------------------------------- */
- (id)init
{
    self = [super init];
    if (self == nil)
    {
        return nil;
    }
    
    _texture = nil;
    _viewport = {};
    _orthogonalProjectionMatrix = nil;
    _mtkView = nil;
    _renderPassDescriptor = nil;
    
    return self;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      サイズで初期化
 *  \param[in]  size            サイズ
 *  \param[in]  pixelFormat     ピクセルフォーマット
 *  \param[in]  device          Metalデバイス
 *  \param[in]  commandQueue    コマンドキュー
 *  \return     自身のクラス
 */
/* ------------------------------------------------------------------------- */
- (id)initWithSize:(simd_uint2)size
       pixelFormat:(MTLPixelFormat)pixelFormat
            device:(id<MTLDevice>)device
      commandQueue:(nonnull id<MTLCommandQueue>)commandQueue
{
    self = [super init];
    if (self == nil)
    {
        return nil;
    }
        
    // サイズで初期化した場合，MetalKitビューは使用しない
    _mtkView = nil;
    
    // ビューポートを設定
    _viewport.originX = 0;
    _viewport.originY = 0;
    _viewport.width = size.x;
    _viewport.height = size.y;
    _viewport.znear = -1.0;
    _viewport.zfar = 1.0;
    
    // ビューポートから平行投影行列を生成
    matrix_float4x4 orthogonalProjectionMatrix;
    orthogonalProjectionMatrix.columns[0] = simd_make_float4(2.0f / (float)_viewport.width, 0.0f, 0.0f, 0.0f);
    orthogonalProjectionMatrix.columns[1] = simd_make_float4(0.0f, 2.0f / -(float)_viewport.height, 0.0f, 0.0f);
    orthogonalProjectionMatrix.columns[2] = simd_make_float4(0.0f, 0.0f, 1.0f, 0.0f);
    orthogonalProjectionMatrix.columns[3] = simd_make_float4(-1.0f, 1.0f, 0.0f, 1.0f);

    // 共有バッファを用意
    id<MTLBuffer> sharedBuffer = [device newBufferWithBytes:&orthogonalProjectionMatrix
                                                     length:sizeof(matrix_float4x4)
                                                    options:MTLResourceStorageModeShared];

    // プライベートバッファを用意
    _orthogonalProjectionMatrix = [device newBufferWithLength:sizeof(matrix_float4x4)
                                            options:MTLResourceStorageModePrivate];

    // 共有バッファからプライベートバッファへ転送するコマンドエンコーダを生成
    id <MTLCommandBuffer> commandBuffer = [commandQueue commandBuffer];
    id <MTLBlitCommandEncoder> blitCommandEncoder = [commandBuffer blitCommandEncoder];
    [blitCommandEncoder copyFromBuffer:sharedBuffer
                          sourceOffset:0
                              toBuffer:_orthogonalProjectionMatrix
                     destinationOffset:0
                                  size:sizeof(matrix_float4x4)];
    [blitCommandEncoder endEncoding];

    // コマンドを実行
    [commandBuffer commit];
    
    // 描画先のテクスチャを生成
    MTLTextureDescriptor *textureDescriptor = [[MTLTextureDescriptor alloc] init];
    textureDescriptor.pixelFormat = pixelFormat;
    textureDescriptor.width = size.x;
    textureDescriptor.height = size.y;
    textureDescriptor.usage = MTLTextureUsageRenderTarget | MTLTextureUsageShaderRead;
    _texture = [device newTextureWithDescriptor:textureDescriptor];
    
    // レンダーパスの記述子を生成
    _renderPassDescriptor = [MTLRenderPassDescriptor renderPassDescriptor];
    _renderPassDescriptor.colorAttachments[0].texture = _texture;
    _renderPassDescriptor.colorAttachments[0].storeAction = MTLStoreActionStore;

    [commandBuffer waitUntilCompleted];

    return self;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      MetalKitビューでパラメータを更新
 *  \param[in]  mtkView                     MetalKitビュー
 *  \param[in]  viewport                    ビューポート
 *  \param[in]  orthogonalProjectionMatrix  平行投影行列
 */
/* ------------------------------------------------------------------------- */
- (void)updateWithMetalKitView:(nonnull MTKView *)mtkView
                      viewport:(nonnull MTLViewport *)viewport
    orthogonalProjectionMatrix:(nonnull id<MTLBuffer>)orthogonalProjectionMatrix
{
    _mtkView = mtkView;
    _texture = mtkView.currentDrawable.texture;
    _viewport = *viewport;
    _orthogonalProjectionMatrix = orthogonalProjectionMatrix;
    _renderPassDescriptor = nil;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      レンダーパスの記述子を取得
 *  \return     レンダーパスの記述子
 */
/* ------------------------------------------------------------------------- */
- (nullable MTLRenderPassDescriptor *)renderPassDescriptor
{
    if (_mtkView != nil)
    {
        return _mtkView.currentRenderPassDescriptor;
    }
    else if (_texture != nil)
    {
        return _renderPassDescriptor;
    }
    
    return nil;
}

@end

#endif  // MGL_RENDERER_ENABLE_METAL
// vim: et ts=4 sw=4 sts=4
