// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_metal_rectangle_renderer.mm
 *  \brief      MGL Metal用矩形レンダラ
 *  \date       Since: December 1, 2020. 19:45:01 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/render/metal/mgl_metal_rectangle_renderer.h>
#if defined(MGL_RENDERER_ENABLE_METAL)

#include <mgl/render/metal/mgl_metal_renderer.h>

namespace MGL::Render::Metal
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
RectangleRenderer::RectangleRenderer() noexcept
    : _attributes(nullptr)
    , _backupAttributesPtr(nullptr)
    , _attributeCount(0)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      描画の実行
 */
/* ------------------------------------------------------------------------- */
void RectangleRenderer::Flush() noexcept
{
    if (_attributeCount == 0)
    {
        return;
    }
    
    [MGLMetalRenderer.sharedInstance drawRectangle:_attributes
                                    attributeCount:_attributeCount];
    
    _attributes = nullptr;
    _attributeCount = 0;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アトリビュートを追加
 *  \param[in]  attribute   追加する描画アトリビュート
 */
/* ------------------------------------------------------------------------- */
void RectangleRenderer::AddAttribute(const RectangleAttribute &attribute) noexcept
{
    // アトリビュートを格納するバッファを用意
    if (_attributes == nullptr)
    {
        _attributes = reinterpret_cast<RectangleAttribute *>([MGLMetalRenderer.sharedInstance reserveDynamicBuffer:sizeof(RectangleAttribute) * kRectangleAttributeMax]);
        
        // レンダラからバッファを予約できなかった場合は自前で用意する
        if (_attributes == nil)
        {
            if (_backupAttributesPtr == nullptr)
            {
                _backupAttributesPtr = STL::make_unique<RectangleAttribute[]>(kRectangleAttributeMax);
            }
            _attributes = _backupAttributesPtr.get();
        }
    }
    
    // アトリビュートをテーブルに追加
    _attributes[_attributeCount] = attribute;
    if (++_attributeCount >= kRectangleAttributeMax)
    {
        Flush();
    }
}
}   // MGL::Render::Metal

#endif  // MGL_RENDERER_ENABLE_METAL
// vim: et ts=4 sw=4 sts=4
