// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_metal_line_renderer.mm
 *  \brief      MGL Metal用ラインレンダラー
 *  \date       Since: December 2, 2020. 3:38:53 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/render/metal/mgl_metal_line_renderer.h>
#if defined(MGL_RENDERER_ENABLE_METAL)

#include <mgl/render/metal/mgl_metal_renderer.h>

namespace MGL::Render::Metal
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
LineRenderer::LineRenderer() noexcept
    : _vertices(nullptr)
    , _backupVerticesPtr(nullptr)
    , _vertexCount(0)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      描画の実行
 */
/* ------------------------------------------------------------------------- */
void LineRenderer::Flush() noexcept
{
    if (_vertexCount == 0)
    {
        return;
    }
    
    [MGLMetalRenderer.sharedInstance drawLine:_vertices
                                  vertexCount:_vertexCount];
    
    _vertices = nullptr;
    _vertexCount = 0;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      描画要素の追加
 *  \param[in]  start   開始座標
 *  \param[in]  end     終了座標
 *  \param[in]  color   色
 */
/* ------------------------------------------------------------------------- */
void LineRenderer::Add(const Vector2 &start, const Vector2 &end, const Color &color) noexcept
{
    // 頂点を格納するバッファを用意
    if (_vertices == nullptr)
    {
        _vertices = reinterpret_cast<LineShaderVertex *>([MGLMetalRenderer.sharedInstance reserveDynamicBuffer:sizeof(LineShaderVertex) * kLineVertexMax]);
        
        // レンダラからバッファを予約できなかった場合は自前で用意する
        if (_vertices == nil)
        {
            if (_backupVerticesPtr == nullptr)
            {
                _backupVerticesPtr = STL::make_unique<LineShaderVertex[]>(kLineVertexMax);
            }
            _vertices = _backupVerticesPtr.get();
        }
    }
    
    // 頂点をテーブルに追加
    _vertices[_vertexCount++] = {{start.x, start.y}, {color.red, color.green, color.blue, color.alpha}};
    _vertices[_vertexCount++] = {{end.x, end.y}, {color.red, color.green, color.blue, color.alpha}};
    
    if (_vertexCount >= kLineVertexMax)
    {
        Flush();
    }
}

}   // namespace MGL::Render::Metal
#endif  // MGL_RENDERER_ENABLE_METAL
// vim: et ts=4 sw=4 sts=4
