// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_metal_dynamic_buffers.mm
 *  \brief      MGL Metal用動的バッファアレイ
 *  \date       Since: February 13, 2021. 16:56:51 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#import <mgl/render/metal/mgl_metal_dynamic_buffers.h>
#if defined(MGL_RENDERER_ENABLE_METAL)

#if defined(MGL_TARGET_MACOS) || defined(TARGET_OS_SIMULATOR)
static const size_t kAlignmentSize = 256;       // macOSのアライメントサイズは256バイト（シミュレータ含む）
#else
static const size_t kAlignmentSize = 4;         // iOS/tvOSのアライメントサイズは4バイト
#endif
static const size_t kAlignmentBit = kAlignmentSize - 1;

@implementation MGLMetalDynamicBuffers
{
    NSArray<id<MTLBuffer>> *_bufferArray;       //!< バッファ配列
    size_t _reserveSize;                        //!< 予約サイズ
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化
 *  \param[in]  bufferCount     生成するバッファの数
 *  \param[in]  length          バッファ1つあたりのサイズ
 *  \param[in]  device          Metalデバイス
 *  \return     失敗時にはnilが返る
 */
/* ------------------------------------------------------------------------- */
- (nullable id)initWithBufferCount:(size_t)bufferCount
                            length:(size_t)length
                            device:(nonnull id<MTLDevice>)device
{
    // 親クラスの初期化
    self = [super init];
    if (self == nil)
    {
        return nil;
    }

    // 引数チェック
    if ((bufferCount == 0) || (length == 0))
    {
        return nil;
    }
    
    // バッファサイズはアライメントして格納
    _bufferSize = length;
    if ((_bufferSize * kAlignmentBit) != 0)
    {
        _bufferSize += kAlignmentBit;
        _bufferSize &= ~kAlignmentBit;
    }
    
    // バッファ配列の生成
    _bufferCount = bufferCount;
    NSMutableArray *mutableBufferArray = [NSMutableArray arrayWithCapacity:_bufferCount];
    for (size_t i = 0; i < _bufferCount; ++i)
    {
        id <MTLBuffer> buffer = [device newBufferWithLength:_bufferSize
                                                    options:MTLResourceStorageModeShared];
        [mutableBufferArray addObject:buffer];
    }
    _bufferArray = [mutableBufferArray copy];
    
    // バッファの数と同じ数を持ったセマフォを生成
    _semaphore = dispatch_semaphore_create((int)_bufferCount);

    // その他のメンバの初期化
    _currentIndex = 0;
    _currentOffset = 0;
    _prevOffset = 0;
    _isReserved = 0;
    _reserveAddress = nil;
    _reserveSize = 0;

    return self;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      次のバッファに切り替え
 */
/* ------------------------------------------------------------------------- */
- (void)next
{
    // セマフォで同期する
    dispatch_semaphore_wait(_semaphore, DISPATCH_TIME_FOREVER);

    // 現在のバッファのインデックスを更新
    if (++_currentIndex >= _bufferCount)
    {
        _currentIndex = 0;
    }
    
    // パラメータを再初期化
    _currentOffset = _prevOffset = 0;
    _reserveSize = 0;
    _reserveAddress = nil;
    _isReserved = false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      現在のバッファを取得
 *  \return     現在のバッファ
 */
/* ------------------------------------------------------------------------- */
- (nonnull id<MTLBuffer>)currentBuffer
{
    return _bufferArray[_currentIndex];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      現在のバッファから指定サイズ分予約
 *  \param[in]  length      予約サイズ
 *  \return     バッファ上の書き込み可能な先頭アドレス．失敗時にはnil
 *  \note       一度予約を行うとcommitするまでreserveとallocateが行えない
 */
/* ------------------------------------------------------------------------- */
- (nullable void *)reserve:(size_t)length
{
    // 予約中であれば失敗
    if (_isReserved)
    {
        return nil;
    }
    
    // アライメント
    if ((length & kAlignmentBit) != 0)
    {
        length += kAlignmentBit;
        length &= ~kAlignmentBit;
    }
    
    // 容量が足りなければ失敗
    if ((_bufferSize - _currentOffset) < length)
    {
        return nil;
    }
    
    _reserveSize = length;
    _isReserved = true;
    _reserveAddress = (uint8_t *)_bufferArray[_currentIndex].contents + _currentOffset;
    
    return _reserveAddress;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      予約したバッファをコミット
 *  \param[in]  length      実際に使用したサイズ
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
- (bool)commit:(size_t)length
{
    // 予約前には実行できない
    if (!_isReserved)
    {
        return false;
    }

    // アライメント
    if ((length & kAlignmentBit) != 0)
    {
        length += kAlignmentBit;
        length &= ~kAlignmentBit;
    }

    // 予約したサイズよりも上回っている場合は失敗
    // NOTE: このチェックに引っかかる場合は範囲外アクセスを行っている可能性があり危険
    if (_reserveSize < length)
    {
        return false;
    }
    
    // パラメータを更新
    _prevOffset = _currentOffset;
    _currentOffset += length;
    _reserveSize = 0;
    _reserveAddress = nil;
    _isReserved = false;
    
    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      現在のバッファからメモリを確保
 *  \param[in]  length  確保するサイズ
 *  \return     バッファ上の書き込み可能な先頭アドレス．失敗時にはnil
 *  \note       同じサイズ分reserveとcommitを行うのと等価
 */
/* ------------------------------------------------------------------------- */
- (nullable void *)allocate:(size_t)length
{
    // 予約中はアロケートできない
    if (_isReserved)
    {
        return nil;
    }
    
    // アライメント
    if ((length & kAlignmentBit) != 0)
    {
        length += kAlignmentBit;
        length &= ~kAlignmentBit;
    }
    
    // 容量が足りなければ失敗
    if ((_bufferSize - _currentOffset) < length)
    {
        return nil;
    }
    
    _prevOffset = _currentOffset;
    _currentOffset += length;
    
    return (uint8_t *)_bufferArray[_currentIndex].contents + _prevOffset;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      現在のバッファの残りサイズを取得
 *  \return     残りサイズ
 */
/* ------------------------------------------------------------------------- */
- (size_t)remainSize
{
    return _bufferSize - _currentOffset;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      指定したアドレスが予約中のアドレスと同じかを調べる
 *  \param[in]  address     チェックするアドレス
 *  \retval     true        同じ
 *  \retval     false       異なる
 */
/* ------------------------------------------------------------------------- */
- (bool)isReservedAddress:(nonnull const void *)address
{
    return (uintptr_t)address == (uintptr_t)_reserveAddress;
}


@end

#endif  // MGL_RENDERER_ENABLE_METAL

// vim: et ts=4 sw=4 sts=4
