// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_indexed_font_resource.cc
 *  \brief      MGL インデックス化対応のフォントリソース基底クラス
 *  \date       Since: April 23, 2023. 10:47:43 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/render/font/mgl_indexed_font_resource.h>

namespace MGL::Render
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  originType      このフォントリソースの原点タイプ
 */
/* ------------------------------------------------------------------------- */
IndexedFontResource::IndexedFontResource(FontOrigin originType) noexcept
    : FontResource(originType)
{
    SetEnabled(FontFeature::IndexedCharacter, true);
    SetEnabled(FontFeature::GetGlyph, true);
    SetEnabled(FontFeature::Tag, true);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          文字の表示
 *  \param[in,out]  workdata    フォント描画のワークデータ
 *  \param[in]      option      表示オプション
 *  \param[in]      text        表示する文字列
 *  \param[in]      formatArgs  フォーマット引数
 *  \retval         true        文字列を最後まで表示した
 *  \retval         false       文字列の表示を途中で中断した
 */
/* ------------------------------------------------------------------------- */
bool IndexedFontResource::Print(
    FontWorkdata &workdata,
    const FontOption &option,
    const char *text,
    const Text::FormatArgs &formatArgs) noexcept
{
    return Print(workdata, option, GetIndexConverter().ToIndexedString(text, true, true).get(), formatArgs);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          インデックス化した文字を表示
 *  \param[in,out]  workdata        フォント描画のワークデータ
 *  \param[in]      option          表示オプション
 *  \param[in]      indexedString   表示する文字列
 *  \param[in]      formatArgs      フォーマット引数
 *  \retval         true            文字列を最後まで表示した
 *  \retval         false           文字列の表示を途中で中断した
 */
/* ------------------------------------------------------------------------- */
bool IndexedFontResource::Print(
    FontWorkdata &workdata,
    const FontOption &option,
    const Text::IndexedCharacter *indexedString,
    const Text::FormatArgs &formatArgs) noexcept
{
    if (indexedString == nullptr)
    {
        return false;
    }

    if (workdata.limitCount == 0)
    {
        return false;
    }

    Renderer2D renderer;

    Work work(renderer);
    work.drawOption = option;
    work.position = workdata.position;
    work.faceType = FontFaceType::Default;
    work.remainCount = workdata.limitCount;

    work.drawOption.SetAlignment(kAlignmentMiddleCenter);

    auto formattedString = Text::Format(indexedString, GetIndexConverter(), formatArgs);

    const Text::IndexedCharacter *current = formattedString.data();
    while (*current != Text::kIndexedCharacterEndOfText)
    {
        // 1行分表示
        current = PrintLine(work, option, current);

        // 改行後の表示位置の更新
        if (*current == MGL::Text::kIndexedCharacterNewLine)
        {
            work.position.x = option.firstPosition.x;
            work.position.y += (GetLineAdvance() * option.scale.y) + option.margin.y;
            current++;
        }

        // 残り表示数が0なら抜ける
        if (work.remainCount == 0)
        {
            break;
        }
    }

    workdata.position = work.position;
    workdata.limitCount = work.remainCount;

    return *current == Text::kIndexedCharacterEndOfText;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          １行分の文字を表示
 *  \param[in,out]  work            表示用パラメータの構造体
 *  \param[in]      fontOption      フォントオプション
 *  \param[in]      indexedString   インデックス化された文字列
 *  \return         表示した最後の文字のアドレス
 */
/* ------------------------------------------------------------------------- */
const Text::IndexedCharacter *IndexedFontResource::PrintLine(
    Work &work,
    const Render::FontOption &fontOption,
    const Text::IndexedCharacter *indexedString) noexcept
{
    // 入力された文字列の先頭が既に行末であれば何もしない
    if ((*indexedString == MGL::Text::kIndexedCharacterNewLine) || (*indexedString == MGL::Text::kIndexedCharacterEndOfText))
    {
        return indexedString;
    }

    // 水平アンカーが左寄せでなければ位置を補正
    if ((fontOption.horizontalAlignment == MGL::Alignment::Horizontal::Center) ||
        (fontOption.horizontalAlignment == MGL::Alignment::Horizontal::Right))
    {
        // 行末までの幅を計算
        float offset = GetLineWidth(work, fontOption, indexedString);

        // 中央揃えの場合はオフセットを半分にする
        if (fontOption.horizontalAlignment == MGL::Alignment::Horizontal::Center)
        {
            offset *= 0.5f;
        }

        work.position.x -= offset;
    }

    // 行の終端まで描画
    for (;; indexedString++)
    {
        auto character = *indexedString;

        if (work.remainCount == 0)
        {
            break;
        }

        // 改行か終了コードで終了
        if ((character == MGL::Text::kIndexedCharacterNewLine) || (character == MGL::Text::kIndexedCharacterEndOfText))
        {
            break;
        }
        // 色の変更
        else if ((character & 0xFF00u) == MGL::Text::kTagParameterColor)
        {
            const MGL::XColor color{static_cast<uint8_t>(character & 0xFFu)};
            work.drawOption.SetMaskColor(color);
        }
        // 書体の変更
        else if ((character & 0xFF00u) == MGL::Text::kTagParameterFace)
        {
            work.faceType = GetFontFaceType(MGL::Text::FaceTypeIndex{static_cast<uint8_t>(character & 0xFFu)});
        }
        // ルビ
        else if ((character == MGL::Text::kIndexedCharacterRubyStart) || (character == MGL::Text::kIndexedCharacterWideRubyStart))
        {
            indexedString = PrintRuby(work, fontOption, indexedString);
        }
        // 全てのタグをリセット
        else if (character == MGL::Text::kIndexedCharacterTagReset)
        {
            work.drawOption.SetMaskColor(fontOption.maskColor);
            work.faceType = FontFaceType::Default;
        }
        // 色をリセット
        else if (character == MGL::Text::kIndexedCharacterColorReset)
        {
            work.drawOption.SetMaskColor(fontOption.maskColor);
        }
        // 書体のリセット
        else if (character == MGL::Text::kIndexedCharacterFaceReset)
        {
            work.faceType = FontFaceType::Default;
        }
        // それ以外はグリフに変換
        else if (const auto *glyph = GetGlyph(character, work.faceType, fontOption); glyph != nullptr)
        {
            if (glyph->texture.GetTexture().IsValid())
            {
                const auto &textureBounds = glyph->texture.GetBounds();
                const MGL::Vector2 drawPosition(
                    work.position.x + (((textureBounds.width / 2.0f) + glyph->bearing.x) * fontOption.scale.x),
                    work.position.y + (((textureBounds.height / 2.0f) - glyph->bearing.y) * fontOption.scale.y));

                work.renderer.DrawSprite(drawPosition, glyph->texture, work.drawOption);
            }

            work.position.x += (fontOption.margin.x + glyph->advance) * fontOption.scale.x;

            if (work.remainCount >= 1)
            {
                work.remainCount--;
            }
        }
    }

    return indexedString;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ルビの表示
 *  \param[in,out]  work            表示用パラメータの構造体
 *  \param[in]      fontOption      フォントオプション
 *  \param[in]      indexedString   インデックス化された文字列
 *  \return         表示した最後の文字のアドレス
 */
/* ------------------------------------------------------------------------- */
const Text::IndexedCharacter *IndexedFontResource::PrintRuby(
    Work &work,
    const FontOption &fontOption,
    const Text::IndexedCharacter *indexedString) noexcept
{
    // 親文字を幅広にするかを取得
    const bool isWide = *indexedString == Text::kIndexedCharacterWideRubyStart;

    // ルビの幅を取得
    float baseTextWidth = 0.0f;
    float rubyTextWidth = 0.0f;
    uint32_t baseTextCount = 0;
    if (const auto *tail = GetRubyWidth(baseTextWidth, rubyTextWidth, baseTextCount, indexedString, work.faceType, fontOption); tail == indexedString)
    {
        return indexedString;
    }

    auto currentPosition = work.position;
    const auto *current = indexedString + 1;

    // 幅広にする場合は追加のマージンを算出
    float additionalMargin = 0.0f;
    if (isWide && (baseTextWidth < rubyTextWidth))
    {
        additionalMargin = (rubyTextWidth - baseTextWidth) / static_cast<float>(baseTextCount + 1);
        baseTextWidth = rubyTextWidth;
        currentPosition.x += additionalMargin;
    }

    bool isSkipped = false;

    // 親文字を描画
    for (; *current != Text::kIndexedCharacterRubyDelimiter; current++)
    {
        if (work.remainCount != 0)
        {
            if (const auto *glyph = GetGlyph(*current, work.faceType, fontOption); glyph != nullptr)
            {
                if (glyph->texture.GetTexture().IsValid())
                {
                    const auto &textureBounds = glyph->texture.GetBounds();
                    const MGL::Vector2 drawPosition(
                        currentPosition.x + (((textureBounds.width / 2.0f) + glyph->bearing.x) * fontOption.scale.x),
                        currentPosition.y + (((textureBounds.height / 2.0f) - glyph->bearing.y) * fontOption.scale.y));

                    work.renderer.DrawSprite(drawPosition, glyph->texture, work.drawOption);
                }

                currentPosition.x += (fontOption.margin.x + glyph->advance) * fontOption.scale.x;
                currentPosition.x += additionalMargin;
            }

            if (work.remainCount >= 1)
            {
                work.remainCount--;
            }
        }
        else
        {
            isSkipped = true;
        }
    }

    // ルビの表示位置を算出
    auto rubyPosition = work.position;
    rubyPosition.x += baseTextWidth / 2.0f;
    rubyPosition.x -= rubyTextWidth / 2.0f;
    rubyPosition.y -= GetRubyOffset();

    // ルビを描画
    for (; *current != Text::kIndexedCharacterRubyEnd; current++)
    {
        if (!isSkipped)
        {
            auto character = *current;

            if (const auto *glyph = GetGlyph(character, FontFaceType::Ruby, fontOption); glyph != nullptr)
            {
                if (glyph->texture.GetTexture().IsValid())
                {
                    const auto &textureBounds = glyph->texture.GetBounds();
                    const MGL::Vector2 drawPosition(
                        rubyPosition.x + (((textureBounds.width / 2.0f) + glyph->bearing.x) * fontOption.scale.x),
                        rubyPosition.y + (((textureBounds.height / 2.0f) - glyph->bearing.y) * fontOption.scale.y));

                    work.renderer.DrawSprite(drawPosition, glyph->texture, work.drawOption);
                }

                rubyPosition.x += glyph->advance * fontOption.scale.x;
            }
        }
    }

    // 送り幅を適用
    work.position.x += baseTextWidth;

    return current;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      1行分の幅を計算
 *  \param[in]  work            表示用パラメータの構造体
 *  \param[in]  fontOption      フォントオプション
 *  \param[in]  indexedString   インデックス化された文字列
 *  \return     1行分の幅
 */
/* ------------------------------------------------------------------------- */
float IndexedFontResource::GetLineWidth(
    const Work &work,
    const FontOption &fontOption,
    const Text::IndexedCharacter *indexedString) noexcept
{
    auto currentFaceType = work.faceType;

    // 行末までの文字数をカウントして補正するオフセット（文字幅）を計算する
    float width = 0.0f;
    for (const auto *current = indexedString;; current++)
    {
        auto character = *current;

        // 改行
        if ((character == MGL::Text::kIndexedCharacterNewLine) || (character == MGL::Text::kIndexedCharacterEndOfText))
        {
            break;
        }
        // 書体の変更
        else if ((character & 0xFF00u) == MGL::Text::kTagParameterFace)
        {
            currentFaceType = GetFontFaceType(MGL::Text::FaceTypeIndex{static_cast<uint8_t>(character & 0xFFu)});
        }
        // ルビ
        else if ((character == MGL::Text::kIndexedCharacterRubyStart) || (character == MGL::Text::kIndexedCharacterWideRubyStart))
        {
            float baseTextWidth = 0.0f;
            float rubyTextWidth = 0.0f;
            uint32_t baseTextCount = 0;
            const auto *next = GetRubyWidth(baseTextWidth, rubyTextWidth, baseTextCount, current, currentFaceType, fontOption);
            if (next != current)
            {
                if (character == Text::kIndexedCharacterWideRubyStart)
                {
                    width += std::max(baseTextWidth, rubyTextWidth);
                }
                else
                {
                    width += baseTextWidth;
                }
                current = next;
            }
        }
        // 書体のリセット
        else if ((character == MGL::Text::kIndexedCharacterTagReset) || (character == MGL::Text::kIndexedCharacterFaceReset))
        {
            currentFaceType = FontFaceType::Default;
        }
        // それ以外はグリフを取得して幅を計算
        else if (const auto *glyph = GetGlyph(character, currentFaceType, fontOption); glyph != nullptr)
        {
            width += (fontOption.margin.x + glyph->advance) * fontOption.scale.x;
        }
    }

    return width;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ルビ付き文字の幅を取得
 */
/* ------------------------------------------------------------------------- */
const Text::IndexedCharacter *IndexedFontResource::GetRubyWidth(
    float &baseTextWidth,
    float &rubyTextWidth,
    uint32_t &baseTextCount,
    const Text::IndexedCharacter *indexedString,
    FontFaceType faceType,
    const FontOption &fontOption) const noexcept
{
    baseTextCount = 0;
    baseTextWidth = 0.0f;
    rubyTextWidth = 0.0f;

    // 親文字の幅を算出
    const auto *current = indexedString + 1;
    for (; *current != Text::kIndexedCharacterRubyDelimiter; current++)
    {
        auto character = *current;

        if (character >= Text::kIndexedCharacterReserveStart)
        {
            return indexedString;
        }
        else if (const auto *glyph = GetGlyph(character, faceType, fontOption); glyph != nullptr)
        {
            baseTextWidth += (fontOption.margin.x + glyph->advance) * fontOption.scale.x;
            baseTextCount++;
        }
    }

    // ルビ文字の幅を算出
    current++;
    for (; *current != Text::kIndexedCharacterRubyEnd; current++)
    {
        auto character = *current;

        if (character >= Text::kIndexedCharacterReserveStart)
        {
            return indexedString;
        }
        else if (const auto *glyph = GetGlyph(character, FontFaceType::Ruby, fontOption); glyph != nullptr)
        {
            rubyTextWidth += (glyph->advance) * fontOption.scale.x;
        }
    }

    return current;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字をインデックス文字に変換
 *  \param[in]  character   変換元の文字（UTF-32）
 *  \param[in]  faceType    フェイスタイプ
 *  \return     インデックス文字．失敗時はMGL::Text::kIndexedCharacterInvalid
 */
/* ------------------------------------------------------------------------- */
Text::IndexedCharacter IndexedFontResource::ToIndexedCharacter(char32_t character, FontFaceType faceType) const noexcept
{
    switch (character)
    {
        // NUL文字は終了コード
        case '\0':
            return Text::kIndexedCharacterEndOfText;

        // ラインフィードは改行
        case 0x0A:
            return Text::kIndexedCharacterNewLine;

        // 復帰コードは無視
        case 0x0D:
            return Text::kIndexedCharacterIgnore;

        default:
            break;
    }

    // 対応するインデックスを取得してインデックス文字に変換
    auto indexedCharacter = static_cast<Text::IndexedCharacter>(GetIndex(character, faceType) & 0xFFFFu);
    if (indexedCharacter >= Text::kIndexedCharacterReserveStart)
    {
        return Text::kIndexedCharacterInvalid;
    }

    return indexedCharacter;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      インデックス文字をUTF-32の文字に変換
 *  \param[in]  indexedCharacter    インデックス文字
 *  \param[in]  faceType            フェイスタイプ
 *  \return     インデックスに対応したUTF-32エンコードの文字．失敗時はNULL文字(0x00)
 */
/* ------------------------------------------------------------------------- */
char32_t IndexedFontResource::ToUTF32(Text::IndexedCharacter indexedCharacter, FontFaceType faceType) const noexcept
{
    if (const auto *glyph = GetGlyph(indexedCharacter, faceType, FontOption()); glyph != nullptr)
    {
        return glyph->code;
    }
    else
    {
        return 0;
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      グリフの取得
 *  \param[in]  character   取得するグリフの文字（UTF-32）
 *  \param[in]  faceType    フェイスタイプ
 *  \param[in]  option      描画オプション
 *  \return     対応するグリフ情報．見つからない場合はnullptr
 */
/* ------------------------------------------------------------------------- */
const FontGlyph *IndexedFontResource::GetGlyph(char32_t character, FontFaceType faceType, const FontOption &option) const noexcept
{
    return GetGlyph(ToIndexedCharacter(character, faceType), faceType, option);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      グリフの取得
 *  \param[in]  character   取得するグリフの文字（インデックス文字）
 *  \param[in]  faceType    フェイスタイプ
 *  \param[in]  option      描画オプション
 *  \return     対応するグリフ情報．見つからない場合はnullptr
 */
/* ------------------------------------------------------------------------- */
const FontGlyph *IndexedFontResource::GetGlyph(Text::IndexedCharacter character, FontFaceType faceType, const FontOption &option) const noexcept
{
    if (character >= Text::kIndexedCharacterReserveStart)
    {
        return nullptr;
    }

    return GetGlyph(static_cast<size_t>(character), faceType, option);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フェイスタイプのインデックスからフォントフェイスを取得
 *  \param[in]  index   フェイスタイプのインデックス
 *  \return     対応するフォントフェイス
 */
/* ------------------------------------------------------------------------- */
FontFaceType IndexedFontResource::GetFontFaceType(Text::FaceTypeIndex index) noexcept
{
    switch (index)
    {
        case Text::FaceTypeIndex::Default: return FontFaceType::Default;
        case Text::FaceTypeIndex::Ruby: return FontFaceType::Ruby;
        case Text::FaceTypeIndex::Bold: return FontFaceType::Bold;
    }

    return FontFaceType::Default;
}

}    // namespace MGL::Render

// vim: et ts=4 sw=4 sts=4
