// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_font_storage.cc
 *  \brief      MGL フォントストレージ
 *  \date       Since: May 26, 2021. 15:05:10 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/render/font/mgl_font_storage.h>

namespace MGL::Render
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      インスタンスの取得
 *  \return     インスタンスの参照
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<FontStorage> &FontStorage::GetInstanceRef() noexcept
{
    static STL::unique_ptr<FontStorage> sInstance = nullptr;
    return sInstance;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フォントリソースの追加
 *  \param[in]  key         フォントキー
 *  \param[in]  resource    追加するフォントリソース
 */
/* ------------------------------------------------------------------------- */
bool FontStorage::Add(FontKey key, const SharedFontResource &resource) noexcept
{
    // 既にキーが使用済みであれば失敗
    if (Get(key) != nullptr)
    {
        return false;
    }

    const std::scoped_lock lock(_mutex);
    _resources.emplace(key, resource);

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フォントリソースの取得
 *  \param[in]  key     フォントキー
 *  \return     キーに対応したフォントリソース．見つからない場合はnullptr
 */
/* ------------------------------------------------------------------------- */
SharedFontResource FontStorage::Get(FontKey key) noexcept
{
    const std::scoped_lock lock(_mutex);

    const auto it = _resources.find(key);
    if (it == _resources.end())
    {
        return nullptr;
    }

    return it->second;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フォントリソースの削除
 *  \param[in]  key     フォントキー
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool FontStorage::Remove(FontKey key) noexcept
{
    const std::scoped_lock lock(_mutex);

    const auto it = _resources.find(key);
    if (it == _resources.end())
    {
        return false;
    }

    _resources.erase(it);

    return true;
}
}    // namespace MGL::Render

// vim: et ts=4 sw=4 sts=4
